<?php
/*
Plugin Name: CalDAV Calendar Importer
Description: Imports events from CalDAV calendars using a REPORT query (Nextcloud compatible).
Version: 8.2.1
Author: vanMeerdervoort
*/

if (!defined('ABSPATH')) exit; // Exit if accessed directly

// --- LOGFUNCTIE ---
function sci_log($msg) {
	static $logfile = null;
	if ($logfile === null) {
		$logfile = plugin_dir_path(__FILE__) . 'sci_caldav.log';
		if (!file_exists($logfile)) file_put_contents($logfile, '');
	}
	error_log("[" . date('Y-m-d H:i:s') . "] $msg\n", 3, $logfile);
	if (isset($GLOBALS['sci_live_log'])) {
		$GLOBALS['sci_live_log'] .= $msg . "\n";
	}
}

// --- DB SETUP ---
function sci_activate() {
	global $wpdb;
	$charset = $wpdb->get_charset_collate();

	$wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}sci_events;");
	$wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}sci_calendars;");

	$wpdb->query("CREATE TABLE {$wpdb->prefix}sci_calendars (
        id INT AUTO_INCREMENT PRIMARY KEY,
        url TEXT NOT NULL,
        color VARCHAR(7) NOT NULL,
        username VARCHAR(100),
        password VARCHAR(100),
        website TEXT
    ) $charset;");

	$wpdb->query("CREATE TABLE {$wpdb->prefix}sci_events (
        id BIGINT AUTO_INCREMENT PRIMARY KEY,
        calendar_id INT NOT NULL,
        uid VARCHAR(255),
        title TEXT,
        start DATETIME,
        location TEXT,
        description TEXT,
        color VARCHAR(7),
        last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (calendar_id) REFERENCES {$wpdb->prefix}sci_calendars(id) ON DELETE CASCADE
    ) $charset;");
}

function sci_deactivate() {
	global $wpdb;
	$wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}sci_events;");
	$wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}sci_calendars;");
}

register_activation_hook(__FILE__, 'sci_activate');
register_deactivation_hook(__FILE__, 'sci_deactivate');

// --- ADMIN MENU ---
add_action('admin_menu', function () {
	add_menu_page('CalDAV Importer', 'CalDAV Importer', 'manage_options', 'sci_calendar_importer', 'sci_admin_page');
});

// --- SETTINGS REGISTRATIE ---
add_action('admin_init', function() {
	register_setting('sci_options', 'sci_show_info_button');
	register_setting('sci_options', 'sci_random_event_colors');
	register_setting('sci_options', 'sci_color_palette', [
		'sanitize_callback' => function ($value) {
			if (!is_array($value)) return [];
			return array_values(array_filter(array_map('sanitize_hex_color', $value)));
		}
	]);
	register_setting('sci_options', 'sci_verbose_sync');
});

// --- DESCRIPTION LINKS PARSER ---
function sci_parse_description_links($text) {
	sci_log("Parse functie aangeroepen met text: " . substr($text, 0, 200));
	
	$result = [
		'ticket_url' => null,
		'info_url' => null,
		'description_text' => null,
		'soldout' => false,
	];

	if (empty($text)) {
		return $result;
	}

	// Detecteer SOLDOUT (case-insensitive, overal in tekst)
	if (preg_match('/\bSOLDOUT\b/i', $text)) {
		$result['soldout'] = true;
		// Verwijder SOLDOUT tag uit tekst
		$text = preg_replace('/\bSOLDOUT\b/i', '', $text);
	}

	// Normaliseer newlines eerst - verwijder alle varianten
	$text = preg_replace('/\r\n|\r|\n/', ' ', $text);
	$text = preg_replace('/\\\\n|\\\\r\\\\n/', ' ', $text); // Verwijder letterlijke \n strings
	$text = preg_replace('/\s+/', ' ', $text); // Meerdere spaties naar één spatie
	$text = trim($text);

	// Parse velden met betere regex die stopt bij volgende veldnaam
	// Extract Tickets URL - pak alles vanaf https:// tot het volgende veld label of einde
	if (preg_match('/Tickets:\s*(https?:\/\/.*?)(?=\s+(?:Info|Description|Tickets):|$)/i', $text, $matches)) {
		$url = trim($matches[1]);
		// Verwijder eventuele trailing veldnamen die per ongeluk zijn meegenomen
		$url = preg_replace('/\s+(Info|Description|Tickets):.*/i', '', $url);
		// Verwijder alle spaties uit URL (URLs hebben geen spaties - dit pakt ook woorden na de URL op)
		$url = preg_replace('/\s+/', '', $url);
		$url = trim($url);
		if (!empty($url) && filter_var($url, FILTER_VALIDATE_URL)) {
			$result['ticket_url'] = esc_url($url);
		}
		// Verwijder Tickets veld uit tekst voor verdere verwerking
		$text = preg_replace('/Tickets:\s*.*?(?=\s*(?:Info|Description|Tickets):|$)/is', '', $text);
	}

	// Extract Info URL - pak alles vanaf https:// tot het volgende veld label of einde
	if (preg_match('/Info:\s*(https?:\/\/.*?)(?=\s+(?:Tickets|Description|Info):|$)/i', $text, $matches)) {
		$url = trim($matches[1]);
		// Verwijder eventuele trailing veldnamen die per ongeluk zijn meegenomen
		$url = preg_replace('/\s+(Tickets|Description|Info):.*/i', '', $url);
		// Verwijder alle spaties uit URL (URLs hebben geen spaties - dit pakt ook woorden na de URL op)
		$url = preg_replace('/\s+/', '', $url);
		$url = trim($url);
		if (!empty($url) && filter_var($url, FILTER_VALIDATE_URL)) {
			$result['info_url'] = esc_url($url);
		}
		// Verwijder Info veld uit tekst voor verdere verwerking
		$text = preg_replace('/Info:\s*.*?(?=\s*(?:Tickets|Description|Info):|$)/is', '', $text);
	}

	// Extract Description tekst
	if (preg_match('/Description:\s*(.*?)(?=\s*(?:Tickets|Info):|$)/is', $text, $matches)) {
		$desc_text = trim($matches[1]);
		// Verwijder eventuele trailing veldnamen
		$desc_text = preg_replace('/\s+(Tickets|Info|Description):.*/i', '', $desc_text);
		$desc_text = trim($desc_text);
		if (!empty($desc_text)) {
			$result['description_text'] = $desc_text;
		}
		// Verwijder Description veld uit tekst
		$text = preg_replace('/Description:\s*.*?(?=\s*(?:Tickets|Info):|$)/is', '', $text);
	}

	// Als er nog tekst over is zonder veldlabel, gebruik dat als description
	$remaining_text = trim($text);
	// Verwijder eventuele resterende veldnamen
	$remaining_text = preg_replace('/\b(Tickets|Info|Description):\s*/i', '', $remaining_text);
	$remaining_text = trim($remaining_text);
	
	if (!empty($remaining_text) && empty($result['description_text'])) {
		$result['description_text'] = $remaining_text;
	}

	// Cleanup: verwijder CalDAV syntax uit description_text
	if (!empty($result['description_text'])) {
		$result['description_text'] = preg_replace('/END:(VALARM|VEVENT).*/is', '', $result['description_text']);
		$result['description_text'] = preg_replace('/BEGIN:(VALARM|VEVENT).*/is', '', $result['description_text']);
		$result['description_text'] = preg_replace('/Reminder.*/is', '', $result['description_text']);
		// Verwijder DTSTART en DTEND velden (met of zonder parameters, zoals DTSTART;TZID=Europe/Copenhagen:20260614T150000)
		// Pattern: DTSTART (optioneel met parameters zoals ;TZID=Europe/Copenhagen) : waarde (alles tot spatie/einde)
		$result['description_text'] = preg_replace('/\bDTSTART(?:;[^:]+)?:\s*\S+(?:\s+|$)/i', '', $result['description_text']);
		$result['description_text'] = preg_replace('/\bDTEND(?:;[^:]+)?:\s*\S+(?:\s+|$)/i', '', $result['description_text']);
		$result['description_text'] = preg_replace('/\s+/', ' ', $result['description_text']);
		$result['description_text'] = trim($result['description_text']);
	}

	return $result;
}


// --- SYNC FUNCTIE ---
function sci_sync_events($verbose=false) {
	global $wpdb;
	$cal_table = $wpdb->prefix.'sci_calendars';
	$evt_table = $wpdb->prefix.'sci_events';
	$cals = $wpdb->get_results("SELECT * FROM $cal_table");
	foreach ($cals as $cal) {
		$url = rtrim($cal->url, '/') . '/';
		sci_log("Fetching: $url");
		$xml='<?xml version="1.0" encoding="utf-8" ?>
<C:calendar-query xmlns:D="DAV:" xmlns:C="urn:ietf:params:xml:ns:caldav">
  <D:prop><D:getetag/><C:calendar-data/></D:prop>
  <C:filter><C:comp-filter name="VCALENDAR"><C:comp-filter name="VEVENT"/></C:comp-filter></C:filter>
</C:calendar-query>';
		$ch = curl_init($url);

		if ($verbose) sci_log("cURL: init gedaan");

		curl_setopt_array($ch, [
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_USERPWD => "$cal->username:$cal->password",
			CURLOPT_HTTPHEADER => ["Content-Type:application/xml", "Depth:1"],
			CURLOPT_CUSTOMREQUEST => "REPORT",
			CURLOPT_POSTFIELDS => $xml
		]);
		$resp = curl_exec($ch);

		if ($resp === false) {
			$err = curl_error($ch);
			sci_log("cURL ERROR: $err");
		} else {
			sci_log("cURL: curl_exec succesvol, response lengte: " . strlen($resp));
		}

		$code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		curl_close($ch);
		
		if ($code !== 207) {
			if ($verbose) sci_log("❌ CalDAV REPORT failed: HTTP $code");
			continue;
		}
		
		if ($verbose) sci_log("✅ CalDAV REPORT success: HTTP $code");

		preg_match_all('/BEGIN:VEVENT.*?END:VEVENT/s', $resp, $m);
		$parsed = [];
		
		foreach ($m[0] as $ve) {
			preg_match('/UID:(.+)/', $ve, $u);
			preg_match('/DTSTART[^:]*:([^\s]+)/', $ve, $d);
			
			if (empty($u[1]) || empty($d[1])) continue;
			
			preg_match('/SUMMARY:(.+)/', $ve, $s);
			preg_match('/LOCATION:(.*)/', $ve, $l);
			preg_match('/^DESCRIPTION:(.*?)(?=\n(?:BEGIN:VALARM|END:VEVENT|[A-Z]+:))/sm', $ve, $desc);

			if (!empty($desc[1])) {
				if ($verbose) {
					sci_log("RAW DESCRIPTION: " . var_export(trim($desc[1]), true));
				}
				
				// Normaliseer alle newline varianten: verwijder \r\n, \n, \\n (letterlijke backslash+n), en whitespace na newlines
				$desc[1] = preg_replace('/\r\n|\r|\n/', ' ', $desc[1]);
				$desc[1] = preg_replace('/\\\\n|\\\\r\\\\n/', ' ', $desc[1]); // Verwijder letterlijke \n strings
				
				// Verwijder CalDAV technische velden die in description kunnen voorkomen
				// Pattern: DTSTART/DTEND (optioneel met parameters zoals ;TZID=Europe/Copenhagen) : waarde (alles tot spatie/einde)
				$desc[1] = preg_replace('/\bDTSTART(?:;[^:]+)?:\s*\S+(?:\s+|$)/i', '', $desc[1]);
				$desc[1] = preg_replace('/\bDTEND(?:;[^:]+)?:\s*\S+(?:\s+|$)/i', '', $desc[1]);
				$desc[1] = preg_replace('/END:(VALARM|VEVENT).*/is', '', $desc[1]);
				$desc[1] = preg_replace('/BEGIN:(VALARM|VEVENT).*/is', '', $desc[1]);
				$desc[1] = preg_replace('/Reminder.*/is', '', $desc[1]);
				
				$desc[1] = preg_replace('/\s+/', ' ', $desc[1]); // Meerdere spaties naar één spatie
				$desc[1] = trim($desc[1]);
			}

			$parsed[] = array(
				'uid' => trim($u[1]),
				'title' => trim($s[1] ?? 'Untitled'),
				'start' => date('Y-m-d H:i:s', strtotime($d[1])),
				'location' => trim($l[1] ?? ''),
				'description' => trim($desc[1] ?? ''),
				'color' => $cal->color,
				'calendar_id' => $cal->id
			);
		}
		
		$existing = $wpdb->get_col($wpdb->prepare("SELECT uid FROM $evt_table WHERE calendar_id=%d", $cal->id));
		$uids = wp_list_pluck($parsed, 'uid');
		
		foreach ($parsed as $e) {
			sci_upsert_event($e['calendar_id'], $e, $verbose);
		}
		
		foreach ($existing as $uid) {
			if (!in_array($uid, $uids)) {
				$wpdb->delete($evt_table, ['calendar_id' => $cal->id, 'uid' => $uid]);
				if ($verbose) sci_log("Deleted $uid");
			}
		}
	}
}
function sci_upsert_event($cid, $d, $verbose = false) {
	global $wpdb;
	$t = $wpdb->prefix . 'sci_events';
	$ex = $wpdb->get_row($wpdb->prepare("SELECT id,title,start,location,description,color FROM $t WHERE calendar_id=%d AND uid=%s", $cid, $d['uid']));
	if ($ex) {
		if (
			$ex->title !== $d['title'] ||
			$ex->start !== $d['start'] ||
			$ex->location !== $d['location'] ||
			$ex->description !== $d['description'] ||
			$ex->color !== $d['color']
		) {
			$wpdb->update($t, [
				'title' => $d['title'],
				'start' => $d['start'],
				'location' => $d['location'],
				'description' => $d['description'],
				'color' => $d['color']
			], ['id' => $ex->id]);
			if($verbose) sci_log("Updated {$d['uid']}");
		}
	} else {
		$wpdb->insert($t, [
			'calendar_id' => $cid,
			'uid' => $d['uid'],
			'title' => $d['title'],
			'start' => $d['start'],
			'location' => $d['location'],
			'description' => $d['description'],
			'color' => $d['color']
		]);
		if($verbose) sci_log("Inserted {$d['uid']}");
	}
}

// --- ADMIN PAGINA ---
function sci_admin_page() {
	global $wpdb;
	$table = $wpdb->prefix . 'sci_calendars';

	echo '<div class="wrap"><h1>CalDAV Importer</h1>';

	// ============ FORM 1: Opties opslaan via options.php ============
	echo '<form method="post" action="options.php">';
	settings_fields('sci_options');
	do_settings_sections('sci_options'); // optioneel
	echo '<div class="sci-admin-toolbar">';
	echo '<label style="margin-right: 2em;"><input type="checkbox" name="sci_show_info_button" value="1" ' . checked(1, get_option('sci_show_info_button'), false) . '> Toon Info-knop</label>';
	echo '<label style="margin-right: 2em;"><input type="checkbox" name="sci_verbose_sync" value="1" ' . checked(1, get_option('sci_verbose_sync'), false) . '> Verbose logging</label>';
	echo '<label style="margin-right: 2em;"><input type="checkbox" name="sci_random_event_colors" value="1" ' . checked(1, get_option('sci_random_event_colors'), false) . '> Willekeurige event-kleuren</label>';
	echo '<div style="margin-left:auto;">';
	submit_button('Instellingen opslaan', 'primary', 'submit', false);
	echo '</div>';
	echo '</div>'; // einde .sci-admin-toolbar

	// Kleureninstellingen
	echo '<div class="sci-color-settings" style="margin-top:20px;">';
	echo "<br /><br />";
	echo '<div style="margin-top:1em;"><h3>Kleurenpalet</h3>';
	echo "<br />";
	$palette = get_option('sci_color_palette', []);
	if (!is_array($palette)) $palette = [];
	echo '<div id="sci_palette_container">';
	foreach ($palette as $color) {
		echo '<div class="sci-color-row">
            <input type="text" name="sci_color_palette[]" class="sci-color-picker" value="' . esc_attr($color) . '">
            <button type="button" class="button sci-remove-color">×</button>
        </div>';
	}
	echo '</div>';
	echo '<button type="button" class="button" id="sci_add_color" style="margin-top:25px">+ Voeg kleur toe</button>';
	echo "<br /><br />";
	echo '</div>';
	echo '</div>'; // einde .sci-color-settings
	echo '</form>'; // einde form 1

	// ============ FORM 2: SYNC NOW =============
	echo '<form method="post" style="display:inline; margin-top: 1em;">';
	echo '<hr style="margin-bottom:16px; margin-top:16px;">';
	wp_nonce_field('sci_sync_nonce', 'sci_sync_nonce_field');
	echo '<button type="submit" name="sci_sync_now" class="button">Sync Now</button>';
	echo '</form>';

	// ============ SYNC HANDLER ============
	if ( isset($_POST['sci_sync_now']) &&
	     isset($_POST['sci_sync_nonce_field']) &&
	     check_admin_referer('sci_sync_nonce', 'sci_sync_nonce_field')) {
		$GLOBALS['sci_live_log'] = '';
		$verbose = get_option('sci_verbose_sync') ? true : false;
		sci_log("🔄 Manual sync triggered by user. Verbose: " . ($verbose ? 'ON' : 'OFF'));
		sci_sync_events($verbose);
	}

	// ==== ADD/DELETE/UPDATE Calendar (deze handeling kan blijven zoals in jouw code) ====
	if (isset($_POST['sci_add_calendar'])) {
		$wpdb->insert($table, [
			'url' => sanitize_text_field($_POST['calendar_url']),
			'color' => sanitize_hex_color($_POST['calendar_color']),
			'username' => sanitize_text_field($_POST['calendar_username']),
			'password' => sanitize_text_field($_POST['calendar_password']),
			'website' => esc_url_raw($_POST['calendar_website']),
		]);
	}
	if (isset($_POST['sci_delete_calendar'])) {
		$wpdb->delete($table, ['id' => intval($_POST['calendar_id'])]);
	}
	if (isset($_POST['sci_update_calendar'])) {
		$wpdb->update($table, [
			'url' => sanitize_text_field($_POST['calendar_url']),
			'color' => sanitize_hex_color($_POST['calendar_color']),
			'username' => sanitize_text_field($_POST['calendar_username']),
			'password' => sanitize_text_field($_POST['calendar_password']),
			'website' => esc_url_raw($_POST['calendar_website']),
		], ['id' => intval($_POST['calendar_id'])]);
	}

	// ==== Add Calendar Form ====
	echo '<div id="add_calendar_form" style="display:none;">
        <h3>Add Calendar</h3>
        <form method="post">
            <table class="form-table">
                <tr><th>Calendar URL:</th><td><input type="text" name="calendar_url" class="regular-text"></td></tr>
                <tr><th>Color:</th><td><input type="text" name="calendar_color" class="sci-color-picker" value="#ff0000"></td></tr>
                <tr><th>Username:</th><td><input type="text" name="calendar_username" class="regular-text"></td></tr>
                <tr><th>Password:</th><td><input type="password" name="calendar_password" class="regular-text"></td></tr>
                <tr><th>Website:</th><td><input type="text" name="calendar_website" class="regular-text"></td></tr>
            </table>
            <p><input type="submit" name="sci_add_calendar" class="button-primary" value="Add Calendar"></p>
        </form>
    </div>';

	echo '<button id="add_calendar_btn" class="button-primary" style="margin-bottom: 15px;">Add Calendar</button>';
	echo '<h2>Saved Calendars</h2><table class="wp-list-table widefat striped">';

	// ==== List Calendars ====
	$calendars = $wpdb->get_results("SELECT * FROM $table");
	echo '<h2>Saved Calendars</h2><table class="wp-list-table widefat striped">
        <thead><tr><th>URL</th><th>Color</th><th>Website</th><th>Actions</th></tr></thead><tbody>';
	foreach ($calendars as $cal) {
		echo '<tr>';
		echo '<td>' . esc_html($cal->url) . '</td>';
		echo '<td><span style="
            display: inline-block;
            padding: 3px 8px;
            background-color: ' . esc_attr($cal->color) . ';
            color: #fff;
            border-radius: 3px;
            font-family: monospace;
        ">' . esc_html($cal->color) . '</span></td>';
		echo '<td><a href="' . esc_url($cal->website) . '" target="_blank">Visit</a></td>';
		echo '<td>';
		echo '<form method="post" style="display:inline;">'
		     . '<input type="hidden" name="calendar_id" value="' . esc_attr($cal->id) . '">'
		     . '<input type="submit" name="sci_delete_calendar" class="button-secondary" value="Delete">'
		     . '</form> ';
		echo '<button class="button-secondary edit_calendar_btn" data-calendar-id="' . esc_attr($cal->id) . '">Edit</button>';
		echo '<div id="edit_form_' . esc_attr($cal->id) . '" class="edit_calendar_form" style="display:none;">
            <h3>Edit Calendar</h3>
            <form method="post">
                <input type="hidden" name="calendar_id" value="' . esc_attr($cal->id) . '">
                <table class="form-table">
                    <tr><th>Calendar URL:</th><td><input type="text" name="calendar_url" class="regular-text" value="' . esc_attr($cal->url) . '"></td></tr>
                    <tr><th>Color:</th><td><input type="text" name="calendar_color" class="sci-color-picker" value="' . esc_attr($cal->color) . '"></td></tr>
                    <tr><th>Username:</th><td><input type="text" name="calendar_username" class="regular-text" value="' . esc_attr($cal->username) . '"></td></tr>
                    <tr><th>Password:</th><td><input type="password" name="calendar_password" class="regular-text" value="' . esc_attr($cal->password) . '"></td></tr>
                    <tr><th>Website:</th><td><input type="text" name="calendar_website" class="regular-text" value="' . esc_attr($cal->website) . '"></td></tr>
                </table>
                <p><input type="submit" name="sci_update_calendar" class="button-primary" value="Update Calendar"></p>
            </form>
        </div>';
		echo '</td></tr>';
	}
	echo '</tbody></table></div>';

	// ==== LOGFILE TONEN ====
	$logfile = plugin_dir_path(__FILE__) . 'sci_caldav.log';
	$log = '';
	if (file_exists($logfile)) {
		$log = file_get_contents($logfile);
	}
	echo '<h2>Logbestand</h2>';
	echo '<form method="post"><textarea id="sci_log_viewer" rows="18" style="width:100%;font-family:monospace;" readonly>' . esc_textarea($log) . '</textarea><br>';
	echo '<button type="submit" name="sci_refresh_log" class="button">Log verversen</button></form>';
	echo '<script>
        jQuery(function($){
            $("form").on("submit", function(){
                setTimeout(function(){
                    var ta = document.getElementById("sci_log_viewer");
                    if (ta) ta.scrollTop = ta.scrollHeight;
                }, 100);
            });
        });
    </script>';

	echo '<script>
        jQuery(document).ready(function($){
            $("#add_calendar_btn").click(function(){ $("#add_calendar_form").toggle(); });
            $(".edit_calendar_btn").click(function(){ var id=$(this).data("calendar-id"); $("#edit_form_"+id).toggle(); });
            $(document).click(function(e){
                if(!$(e.target).closest("#add_calendar_form,#add_calendar_btn").length) $("#add_calendar_form").hide();
                $(".edit_calendar_form").each(function(){ if(!$(e.target).closest(this).length && !$(e.target).closest(".edit_calendar_btn").length) $(this).hide(); });
            });
        });
    </script>';
	echo '<script>
    jQuery(function($){
        function initColorPickers() {
            $(".sci-color-picker").wpColorPicker();
        }
        $("#sci_add_color").on("click", function() {
            var row = $("<div class=\'sci-color-row\'>"+
                "<input type=\'text\' name=\'sci_color_palette[]\' class=\'sci-color-picker\' value=\'#000000\'>"+
                "<button type=\'button\' class=\'button sci-remove-color\'>×</button></div>");
            $("#sci_palette_container").append(row);
            row.find(".sci-color-picker").wpColorPicker();
        });
        $(document).on("click", ".sci-remove-color", function(){
            $(this).closest(".sci-color-row").remove();
        });
        initColorPickers();
    });
    </script>
    <style>
    .sci-color-row { display: flex; align-items: center; margin-bottom: 8px; }
    .sci-color-row .sci-color-picker { margin-right: 8px; width: 100px; }
    </style>';
}

// --- SHORTCODE ---
add_shortcode('sci_events', function () {
	global $wpdb;
	$table = $wpdb->prefix . 'sci_events';
	$calendar_table = $wpdb->prefix . 'sci_calendars';
	$events = $wpdb->get_results("SELECT e.*, c.url as calendar_url, c.color as calendar_color, c.website as calendar_website FROM $table e JOIN $calendar_table c ON e.calendar_id = c.id WHERE e.start >= NOW() ORDER BY e.start ASC");

	if (!$events) return 'No events available.';

	$output = '<div class="sci-events-list">';
	foreach ($events as $event) {
		$date = new DateTime($event->start);
		$calendar_name = ucwords(str_replace('-', ' ', basename(trim($event->calendar_url, '/'))));
		$calendar_class = strtolower(str_replace(['-', ' '], '', basename(trim($event->calendar_url, '/'))));
		$palette = get_option('sci_color_palette', []);
		$use_random = get_option('sci_random_event_colors');
		static $last_color = null;

		$calendar_color = esc_attr($event->calendar_color);

		if ($use_random && is_array($palette) && count($palette) > 0) {
			$choices = array_values(array_filter($palette, fn($c) => $c !== $last_color));
			if (empty($choices)) $choices = $palette;
			$calendar_color = esc_attr($choices[array_rand($choices)]);
			$last_color = $calendar_color;
		}

		$calendar_website = esc_url($event->calendar_website);
		$links = sci_parse_description_links($event->description);

		$output .= '<div class="sci-event sci-calendar-' . esc_attr($calendar_class) . '">';
		$output .= '<div class="sci-date-col" style="border-left-color:' . $calendar_color . ';">';
		$output .= '<div class="sci-date">' . esc_html($date->format('d F')) . '</div>';
		$output .= '<div class="sci-year">' . esc_html($date->format('Y')) . '</div>';
		$output .= '</div>';
		$output .= '<div class="sci-info-col">';
		$output .= '<div class="sci-calendar-name" style="color:' . $calendar_color . ';">' . esc_html($calendar_name) . '</div>';
		$output .= '<div class="sci-title">';
		$output .= '<strong>' . esc_html($event->title) . '</strong>';
		$output .= '</div>';
		$time = $date->format('H:i:s') !== '00:00:00' ? esc_html($date->format('H:i')) . ' - ' : '';
		$output .= '<div class="sci-location">' . $time . esc_html(str_replace('\\', '', $event->location)) . '</div>';
		if (!empty($links['description_text'])) {
			$output .= '<div class="sci-description">' . esc_html($links['description_text']) . '</div>';
		}
		$output .= '</div>';
		$output .= '<div class="sci-website-col">';
		if (!empty($links['info_url'])) {
			// Toon alleen als de info URL NIET exact de homepage is
			$current_site = rtrim(home_url(), '/');
			$info_site = rtrim($links['info_url'], '/');

			// Parse beide URLs om domein en pad te vergelijken
			$current_parse = parse_url($current_site);
			$info_parse = parse_url($info_site);

			// Verberg alleen als het exact dezelfde URL is (zelfde domein EN zelfde pad)
			$is_same = ($current_parse['host'] === $info_parse['host'] &&
			            ($info_parse['path'] ?? '/') === ($current_parse['path'] ?? '/'));

			if (!$is_same) {
				$output .= '<a href="' . $links['info_url'] . '" class="sci-website-button" target="_blank">Meer Info</a><br>';
			}
		} elseif (get_option('sci_show_info_button') && !empty($event->calendar_website)) {
			$current_site = rtrim(home_url(), '/');
			$calendar_site = rtrim($event->calendar_website, '/');

			$current_parse = parse_url($current_site);
			$calendar_parse = parse_url($calendar_site);

			$is_same = ($current_parse['host'] === $calendar_parse['host'] &&
			            ($calendar_parse['path'] ?? '/') === ($current_parse['path'] ?? '/'));

			if (!$is_same) {
				$output .= '<a href="' . esc_url($event->calendar_website) . '" class="sci-website-button" target="_blank">Meer Info</a><br>';
			}
		}
		if (!empty($links['soldout'])) {
			$output .= '<span class="sci-soldout-badge">Uitverkocht</span>';
		} elseif ($links['ticket_url']) {
			$output .= '<a href="' . $links['ticket_url'] . '" class="sci-website-button sci-ticket-button" target="_blank">Tickets</a>';
		}
		$output .= '</div>';
		$output .= '</div>';
	}
	$output .= '</div>';
	return $output;
});

// --- SCRIPTS/CRON/UPDATES ---
add_action('wp_enqueue_scripts', function () {
	wp_enqueue_style('sci_calendar_styles', plugin_dir_url(__FILE__) . 'style.css');
});
if (!wp_next_scheduled('sci_sync_hourly')) {
	wp_schedule_event(time(), 'hourly', 'sci_sync_hourly');
}
add_action('sci_sync_hourly', function() { sci_sync_events(false); }); // Cron: altijd zonder verbose

add_action('admin_enqueue_scripts', function ($hook) {
	if (!in_array($hook, ['toplevel_page_sci_calendar_importer', 'settings_page_sci_calendar_settings'])) return;
	wp_enqueue_style('wp-color-picker');
	wp_enqueue_script('wp-color-picker');
	if ($hook === 'toplevel_page_sci_calendar_importer') {
		wp_enqueue_style('sci-admin-css', plugin_dir_url(__FILE__) . 'admin-style.css');
	}
});
add_filter('pre_set_site_transient_update_plugins', function($transient) {
	if (empty($transient->checked)) return $transient;
	$remote = wp_remote_get('https://plugins.nietstezienhier.nl/simple-calendar-importer/update.json', [
		'timeout' => 15,
		'sslverify' => false,
	]);
	if (is_wp_error($remote) || wp_remote_retrieve_response_code($remote) !== 200) return $transient;
	$plugin_data = json_decode(wp_remote_retrieve_body($remote));
	if (!$plugin_data) return $transient;
	$plugin_file = plugin_basename(__FILE__);
	$current_version = $transient->checked[$plugin_file] ?? null;
	if ($current_version && version_compare($plugin_data->version, $current_version, '>')) {
		$transient->response[$plugin_file] = (object)[
			'slug' => $plugin_data->slug,
			'new_version' => $plugin_data->version,
			'url' => $plugin_data->homepage,
			'package' => $plugin_data->download_url,
		];
	}
	return $transient;
});
add_filter('plugins_api', function($result, $action, $args) {
	if ($action !== 'plugin_information' || $args->slug !== 'simple-calendar-importer') return $result;
	$remote = wp_remote_get('https://plugins.nietstezienhier.nl/simple-calendar-importer/update.json', [
		'timeout' => 15,
		'sslverify' => false,
	]);
	if (is_wp_error($remote) || wp_remote_retrieve_response_code($remote) !== 200) return $result;
	$plugin_data = json_decode(wp_remote_retrieve_body($remote));
	if (!$plugin_data) return $result;
	return (object)[
		'name' => $plugin_data->name,
		'slug' => $plugin_data->slug,
		'version' => $plugin_data->version,
		'author' => $plugin_data->author,
		'homepage' => $plugin_data->homepage,
		'sections' => (array) $plugin_data->sections,
		'download_link' => $plugin_data->download_url,
	];
}, 10, 3);
