<?php
/**
 * User Creator Core Functionality
 */

if (!defined('ABSPATH')) exit;

/**
 * Create WordPress user from Mollie Forms registration
 *
 * @param int $form_id Form ID
 * @param object $registration Registration object from database
 * @param array $fields Array of field objects
 * @param bool $skip_enabled_check Skip the enabled check (for bulk import)
 * @return int|WP_Error User ID on success, WP_Error on failure
 */
function mfuc_create_user_from_registration($form_id, $registration, $fields, $skip_enabled_check = false) {
    // Get form settings
    $settings = mfuc_get_form_settings($form_id);
    
    if (!$skip_enabled_check && (!$settings || !isset($settings['enabled']) || !$settings['enabled'])) {
        return new WP_Error('form_not_enabled', __('Formulier is niet geconfigureerd voor gebruikersaanmaak.', 'mollie-forms-user-creator'));
    }
    
    // If no settings but skip check is true, use defaults
    if (!$settings && $skip_enabled_check) {
        $settings = [
            'enabled' => true,
            'user_role' => 'subscriber',
            'create_when' => 'after_payment'
        ];
    }
    
    // Extract email and name from fields
    $email = '';
    $name = '';
    
    foreach ($fields as $field) {
        if ($field->type === 'email') {
            $email = sanitize_email($field->value);
        }
        if ($field->type === 'name') {
            $name = sanitize_text_field($field->value);
        }
    }
    
    // If no email found, try to get from customer table
    if (empty($email)) {
        global $wpdb;
        $mollieForms = new \MollieForms\MollieForms();
        $customer = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$mollieForms->getCustomersTable()} WHERE id = %d",
            $registration->customer_id
        ));
        
        if ($customer && !empty($customer->email)) {
            $email = sanitize_email($customer->email);
        }
        if (empty($name) && $customer && !empty($customer->name)) {
            $name = sanitize_text_field($customer->name);
        }
    }
    
    // Validate email
    if (empty($email) || !is_email($email)) {
        return new WP_Error('invalid_email', __('Geen geldig e-mailadres gevonden in registratie.', 'mollie-forms-user-creator'));
    }
    
    // Check if user already exists
    $existing_user = get_user_by('email', $email);
    
    if ($existing_user) {
        // User exists, update role if needed
        $user_role = isset($settings['user_role']) ? $settings['user_role'] : 'subscriber';
        
        // Only update role if it's different
        if (!in_array($user_role, $existing_user->roles)) {
            $existing_user->set_role($user_role);
        }
        
        // Update user meta with additional fields
        mfuc_update_user_meta_from_fields($existing_user->ID, $fields);
        
        // Log action
        mfuc_log_user_creation($form_id, $registration->id, $existing_user->ID, 'updated');
        
        return $existing_user->ID;
    }
    
    // Create new user
    $username = mfuc_generate_username($email, $name);
    $password = wp_generate_password(12, false);
    
    $user_data = [
        'user_login' => $username,
        'user_email' => $email,
        'user_pass' => $password,
        'display_name' => !empty($name) ? $name : $email,
        'first_name' => mfuc_extract_first_name($name),
        'last_name' => mfuc_extract_last_name($name),
        'role' => isset($settings['user_role']) ? $settings['user_role'] : 'subscriber',
    ];
    
    $user_id = wp_insert_user($user_data);
    
    if (is_wp_error($user_id)) {
        mfuc_log_user_creation($form_id, $registration->id, 0, 'error', $user_id->get_error_message());
        return $user_id;
    }
    
    // Update user meta with additional fields
    mfuc_update_user_meta_from_fields($user_id, $fields);
    
    // Store registration ID in user meta for reference
    update_user_meta($user_id, 'mfuc_registration_id', $registration->id);
    update_user_meta($user_id, 'mfuc_form_id', $form_id);
    
    // Log successful creation
    mfuc_log_user_creation($form_id, $registration->id, $user_id, 'created');
    
    return $user_id;
}

/**
 * Generate unique username from email or name
 */
function mfuc_generate_username($email, $name = '') {
    // Try to use name first
    if (!empty($name)) {
        $username = sanitize_user(str_replace(' ', '', strtolower($name)), true);
        if (!empty($username) && !username_exists($username)) {
            return $username;
        }
    }
    
    // Fall back to email
    $email_parts = explode('@', $email);
    $base_username = sanitize_user($email_parts[0], true);
    $username = $base_username;
    $counter = 1;
    
    while (username_exists($username)) {
        $username = $base_username . $counter;
        $counter++;
    }
    
    return $username;
}

/**
 * Extract first name from full name
 */
function mfuc_extract_first_name($name) {
    if (empty($name)) {
        return '';
    }
    
    $parts = explode(' ', trim($name), 2);
    return $parts[0];
}

/**
 * Extract last name from full name
 */
function mfuc_extract_last_name($name) {
    if (empty($name)) {
        return '';
    }
    
    $parts = explode(' ', trim($name), 2);
    return isset($parts[1]) ? $parts[1] : '';
}

/**
 * Update user meta from registration fields
 */
function mfuc_update_user_meta_from_fields($user_id, $fields) {
    foreach ($fields as $field) {
        // Skip email and name as they're already handled
        if (in_array($field->type, ['email', 'name'])) {
            continue;
        }
        
        // Store field as user meta with prefix
        $meta_key = 'mfuc_' . sanitize_key($field->field);
        update_user_meta($user_id, $meta_key, $field->value);
    }
}

/**
 * Get form settings
 */
function mfuc_get_form_settings($form_id) {
    $all_settings = get_option('mfuc_form_settings', []);
    return isset($all_settings[$form_id]) ? $all_settings[$form_id] : null;
}

/**
 * Save form settings
 */
function mfuc_save_form_settings($form_id, $settings) {
    $all_settings = get_option('mfuc_form_settings', []);
    $all_settings[$form_id] = $settings;
    update_option('mfuc_form_settings', $all_settings);
}

/**
 * Log user creation
 */
function mfuc_log_user_creation($form_id, $registration_id, $user_id, $status = 'created', $notes = '') {
    global $wpdb;
    
    $table = $wpdb->prefix . 'mfuc_user_creation_log';
    
    $wpdb->insert(
        $table,
        [
            'form_id' => $form_id,
            'registration_id' => $registration_id,
            'user_id' => $user_id,
            'status' => $status,
            'notes' => $notes,
        ],
        ['%d', '%d', '%d', '%s', '%s']
    );
}

/**
 * Bulk import users from all registrations of a form
 * Creates WordPress users for all registrations that don't have a user yet
 *
 * @param int $form_id Form ID
 * @return array Results with counts
 */
function mfuc_bulk_import_users($form_id) {
    global $wpdb;
    
    $mollieForms = new \MollieForms\MollieForms();
    $results = [
        'total' => 0,
        'created' => 0,
        'updated' => 0,
        'skipped' => 0,
        'errors' => 0,
        'errors_list' => []
    ];
    
    // Get all registrations for this form
    $registrations = $wpdb->get_results($wpdb->prepare(
        "SELECT * FROM {$mollieForms->getRegistrationsTable()} WHERE post_id = %d ORDER BY id ASC",
        $form_id
    ));
    
    if (empty($registrations)) {
        return $results;
    }
    
    $results['total'] = count($registrations);
    
    // Get form settings (for bulk import, we use settings if available, but don't require enabled)
    $settings = mfuc_get_form_settings($form_id);
    // If no settings exist, use defaults
    if (!$settings) {
        $settings = [
            'enabled' => true,
            'user_role' => 'subscriber',
            'create_when' => 'after_payment'
        ];
    }
    
    foreach ($registrations as $registration) {
        // Check if user already exists for this registration
        $existing_log = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}mfuc_user_creation_log WHERE form_id = %d AND registration_id = %d",
            $form_id,
            $registration->id
        ));
        
        if ($existing_log && $existing_log->user_id > 0) {
            // Check if user still exists
            $user = get_user_by('ID', $existing_log->user_id);
            if ($user) {
                $results['skipped']++;
                continue;
            }
        }
        
        // Get fields for this registration
        $fields = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$mollieForms->getRegistrationFieldsTable()} WHERE registration_id = %d",
            $registration->id
        ));
        
        if (empty($fields)) {
            $results['skipped']++;
            continue;
        }
        
        // Try to create user (skip enabled check for bulk import)
        $user_id = mfuc_create_user_from_registration($form_id, $registration, $fields, true);
        
        if (is_wp_error($user_id)) {
            $results['errors']++;
            $results['errors_list'][] = [
                'registration_id' => $registration->id,
                'error' => $user_id->get_error_message()
            ];
        } else {
            // Check if it was a new user or update
            $log = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}mfuc_user_creation_log WHERE form_id = %d AND registration_id = %d ORDER BY id DESC LIMIT 1",
                $form_id,
                $registration->id
            ));
            
            if ($log && $log->status === 'created') {
                $results['created']++;
            } elseif ($log && $log->status === 'updated') {
                $results['updated']++;
            } else {
                $results['created']++;
            }
        }
    }
    
    return $results;
}


