<?php
/**
 * Mollie Forms Hooks Integration
 */

if (!defined('ABSPATH')) exit;

/**
 * Handle payment paid hook
 * This is the primary hook for creating users after successful payment
 */
add_action('rfmp_payment_paid', 'mfuc_handle_payment_paid', 10, 3);
function mfuc_handle_payment_paid($post_id, $registration, $fields) {
    // Get form settings
    $settings = mfuc_get_form_settings($post_id);
    
    if (!$settings || !isset($settings['enabled']) || !$settings['enabled']) {
        return;
    }
    
    // Check when to create user
    $create_when = isset($settings['create_when']) ? $settings['create_when'] : 'after_payment';
    
    // Only create if setting is 'after_payment' or 'submit_activate_after_payment'
    if ($create_when === 'after_payment' || $create_when === 'submit_activate_after_payment') {
        // For 'submit_activate_after_payment', user might already exist but be inactive
        if ($create_when === 'submit_activate_after_payment') {
            // Check if user was already created
            global $wpdb;
            $mollieForms = new \MollieForms\MollieForms();
            $log = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}mfuc_user_creation_log WHERE form_id = %d AND registration_id = %d",
                $post_id,
                $registration->id
            ));
            
            if ($log && $log->user_id > 0) {
                // User exists, activate it
                $user = get_user_by('ID', $log->user_id);
                if ($user) {
                    // Remove any inactive role or ensure user has the correct role
                    $user_role = isset($settings['user_role']) ? $settings['user_role'] : 'subscriber';
                    $user->set_role($user_role);
                    
                    // Log activation
                    mfuc_log_user_creation($post_id, $registration->id, $user->ID, 'activated');
                }
                return;
            }
        }
        
        // Create user
        mfuc_create_user_from_registration($post_id, $registration, $fields);
    }
}

/**
 * Handle form submitted hook
 * This creates users immediately when form is submitted (if configured)
 * Note: This hook is called before registration is fully saved, so we use a delayed approach
 */
add_action('rfmp_form_submitted', 'mfuc_handle_form_submitted', 10, 2);
function mfuc_handle_form_submitted($post_id, $rfmp_id) {
    // Get form settings
    $settings = mfuc_get_form_settings($post_id);
    
    if (!$settings || !isset($settings['enabled']) || !$settings['enabled']) {
        return;
    }
    
    // Check when to create user
    $create_when = isset($settings['create_when']) ? $settings['create_when'] : 'after_payment';
    
    // Only create if setting is 'on_submit' or 'submit_activate_after_payment'
    if ($create_when === 'on_submit' || $create_when === 'submit_activate_after_payment') {
        // Schedule user creation after registration is saved
        // Use a small delay to ensure registration is in database
        add_action('shutdown', function() use ($post_id, $rfmp_id, $create_when) {
            mfuc_create_user_from_submission($post_id, $rfmp_id, $create_when);
        }, 20);
    }
}

/**
 * Create user from form submission (called after registration is saved)
 */
function mfuc_create_user_from_submission($post_id, $rfmp_id, $create_when) {
    global $wpdb;
    $mollieForms = new \MollieForms\MollieForms();
    
    // Get registration by rfmp_id from payments table
    $payment = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM {$mollieForms->getPaymentsTable()} WHERE rfmp_id = %s",
        $rfmp_id
    ));
    
    if (!$payment) {
        return;
    }
    
    // Get registration
    $registration = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM {$mollieForms->getRegistrationsTable()} WHERE id = %d",
        $payment->registration_id
    ));
    
    if (!$registration) {
        return;
    }
    
    // Get fields
    $fields = $wpdb->get_results($wpdb->prepare(
        "SELECT * FROM {$mollieForms->getRegistrationFieldsTable()} WHERE registration_id = %d",
        $registration->id
    ));
    
    if (empty($fields)) {
        return;
    }
    
    // For 'submit_activate_after_payment', create user but mark as pending
    if ($create_when === 'submit_activate_after_payment') {
        $user_id = mfuc_create_user_from_registration($post_id, $registration, $fields);
        
        if (!is_wp_error($user_id) && $user_id > 0) {
            // Log as pending - will be activated when payment is received
            mfuc_log_user_creation($post_id, $registration->id, $user_id, 'pending_activation');
        }
    } else {
        // Create user immediately
        mfuc_create_user_from_registration($post_id, $registration, $fields);
    }
}

