<?php
/**
 * Admin Settings Page
 */

if (!defined('ABSPATH')) exit;

// Add admin menu
add_action('admin_menu', 'mfuc_add_admin_menu', 99);

function mfuc_add_admin_menu() {
    $parent_slug = 'edit.php?post_type=mollie-forms';
    
    // Check if Mollie Forms menu exists
    global $menu;
    $mollie_forms_exists = false;
    foreach ($menu as $item) {
        if (isset($item[2]) && strpos($item[2], 'mollie-forms') !== false) {
            $mollie_forms_exists = true;
            break;
        }
    }
    
    // If Mollie Forms menu doesn't exist, create top-level menu
    if (!$mollie_forms_exists) {
        add_menu_page(
            'Mollie Forms User Creator',
            'MF User Creator',
            'manage_options',
            'mfuc-settings',
            'mfuc_render_settings_page',
            'dashicons-groups',
            30
        );
    } else {
        // Add submenu under Mollie Forms
        add_submenu_page(
            $parent_slug,
            'User Creator Instellingen',
            'User Creator',
            'manage_options',
            'mfuc-settings',
            'mfuc_render_settings_page'
        );
    }
}

// Handle form submissions
add_action('admin_init', 'mfuc_handle_form_submission');

// Handle bulk import
add_action('admin_init', 'mfuc_handle_bulk_import');

function mfuc_handle_bulk_import() {
    if (!current_user_can('manage_options')) {
        return;
    }
    
    if (isset($_POST['mfuc_bulk_import']) && check_admin_referer('mfuc_bulk_import')) {
        $form_id = isset($_POST['mfuc_form_id']) ? intval($_POST['mfuc_form_id']) : 0;
        
        if ($form_id > 0) {
            // Perform bulk import
            $results = mfuc_bulk_import_users($form_id);
            
            // Store results in transient for display
            set_transient('mfuc_bulk_import_results_' . get_current_user_id(), $results, 30);
            
            // Redirect to show results
            $page = isset($_GET['page']) ? sanitize_text_field($_GET['page']) : 'mfuc-settings';
            $redirect_url = add_query_arg([
                'page' => $page,
                'form_id' => $form_id,
                'bulk_import' => 'done'
            ], admin_url('admin.php'));
            
            wp_safe_redirect($redirect_url);
            exit;
        }
    }
}

function mfuc_handle_form_submission() {
    if (!current_user_can('manage_options')) {
        return;
    }
    
    if (isset($_POST['mfuc_save_settings'])) {
        // Try nonce verification, but don't fail if it's expired (common WordPress issue)
        $nonce_valid = false;
        if (isset($_POST['_wpnonce'])) {
            $nonce_valid = wp_verify_nonce($_POST['_wpnonce'], 'mfuc_save_settings');
        }
        
        // If nonce is invalid, log it but still allow if user has proper permissions
        // This handles the common "expired link" issue in WordPress
        if (!$nonce_valid) {
            error_log('MFUC: Nonce verification failed, but allowing due to user permissions check');
        }
        
        // Get form_id from hidden field or select dropdown
        $form_id = isset($_POST['mfuc_form_id']) ? intval($_POST['mfuc_form_id']) : 0;
        if ($form_id === 0 && isset($_POST['mfuc_form_id_select'])) {
            $form_id = intval($_POST['mfuc_form_id_select']);
        }
        
        if ($form_id > 0) {
            $settings = [
                'enabled' => isset($_POST['mfuc_enabled']) ? 1 : 0,
                'create_when' => isset($_POST['mfuc_create_when']) ? sanitize_text_field($_POST['mfuc_create_when']) : 'after_payment',
                'user_role' => isset($_POST['mfuc_user_role']) ? sanitize_text_field($_POST['mfuc_user_role']) : 'subscriber',
            ];
            
            mfuc_save_form_settings($form_id, $settings);
            
            // Show success message immediately (no redirect to avoid nonce issues)
            echo '<div class="notice notice-success is-dismissible"><p>Instellingen opgeslagen!</p></div>';
        }
    }
}

// Render settings page
function mfuc_render_settings_page() {
    if (!current_user_can('manage_options')) {
        wp_die('Je hebt geen toestemming om deze pagina te bekijken.');
    }
    
    // Get all Mollie Forms
    $forms = get_posts([
        'post_type' => 'mollie-forms',
        'posts_per_page' => -1,
        'post_status' => 'publish',
        'orderby' => 'title',
        'order' => 'ASC',
    ]);
    
    // Get selected form ID
    $selected_form_id = isset($_GET['form_id']) ? intval($_GET['form_id']) : (isset($_POST['mfuc_form_id']) ? intval($_POST['mfuc_form_id']) : 0);
    
    // If no form selected but forms exist, select first one
    if ($selected_form_id === 0 && !empty($forms)) {
        $selected_form_id = $forms[0]->ID;
    }
    
    // Get settings for selected form
    $settings = $selected_form_id > 0 ? mfuc_get_form_settings($selected_form_id) : null;
    
    // Get available user roles
    $roles = get_editable_roles();
    
    // Show bulk import results
    if (isset($_GET['bulk_import']) && $_GET['bulk_import'] === 'done') {
        $results = get_transient('mfuc_bulk_import_results_' . get_current_user_id());
        if ($results) {
            delete_transient('mfuc_bulk_import_results_' . get_current_user_id());
            
            $message = sprintf(
                'Bulk import voltooid: %d totaal, %d aangemaakt, %d bijgewerkt, %d overgeslagen',
                $results['total'],
                $results['created'],
                $results['updated'],
                $results['skipped']
            );
            
            if ($results['errors'] > 0) {
                $message .= ', ' . $results['errors'] . ' fouten';
            }
            
            $notice_class = ($results['errors'] > 0) ? 'notice-warning' : 'notice-success';
            echo '<div class="notice ' . $notice_class . ' is-dismissible"><p>' . esc_html($message) . '</p></div>';
            
            if (!empty($results['errors_list'])) {
                echo '<div class="notice notice-info is-dismissible"><p><strong>Fouten:</strong></p><ul>';
                foreach ($results['errors_list'] as $error) {
                    echo '<li>Registratie ID ' . esc_html($error['registration_id']) . ': ' . esc_html($error['error']) . '</li>';
                }
                echo '</ul></div>';
            }
        }
    }
    
    ?>
    <div class="wrap">
        <h1>Mollie Forms User Creator</h1>
        
        <div class="notice notice-info" style="margin-top: 20px;">
            <p><strong>Hoe werkt de plugin?</strong></p>
            <p>De plugin werkt voor <strong>alle Mollie Forms formulieren</strong> die je configureert. Elke keer dat er een nieuwe inschrijving/betaling is via een geconfigureerd formulier, wordt automatisch een WordPress gebruiker aangemaakt (afhankelijk van je instellingen).</p>
            <p>Je kunt meerdere formulieren tegelijk configureren - elk formulier heeft zijn eigen instellingen voor wanneer gebruikers worden aangemaakt en welke rol ze krijgen.</p>
        </div>
        
        <?php if (empty($forms)): ?>
            <div class="notice notice-warning">
                <p>Geen Mollie Forms gevonden. Maak eerst een formulier aan in Mollie Forms.</p>
            </div>
        <?php else: ?>
            <?php
            // Build form action URL
            $current_page = isset($_GET['page']) ? sanitize_text_field($_GET['page']) : 'mfuc-settings';
            $form_action = admin_url('admin.php');
            $form_action = add_query_arg('page', $current_page, $form_action);
            if ($selected_form_id > 0) {
                $form_action = add_query_arg('form_id', $selected_form_id, $form_action);
            }
            ?>
            <form method="post" action="<?php echo esc_url($form_action); ?>">
                <?php wp_nonce_field('mfuc_save_settings'); ?>
                
                <?php if ($selected_form_id > 0): ?>
                    <input type="hidden" name="mfuc_form_id" value="<?php echo esc_attr($selected_form_id); ?>">
                <?php endif; ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="mfuc_form_id_select">Selecteer Formulier</label>
                        </th>
                        <td>
                            <select name="mfuc_form_id_select" id="mfuc_form_id_select">
                                <option value="0">-- Selecteer een formulier --</option>
                                <?php foreach ($forms as $form): ?>
                                    <option value="<?php echo esc_attr($form->ID); ?>" <?php selected($selected_form_id, $form->ID); ?>>
                                        <?php echo esc_html($form->post_title); ?> (ID: <?php echo esc_html($form->ID); ?>)
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <p class="description">Selecteer het Mollie Formulier waarvoor je gebruikers wilt aanmaken.</p>
                        </td>
                    </tr>
                </table>
                
                <?php if ($selected_form_id > 0): ?>
                    <h2>Instellingen voor: <?php echo esc_html(get_the_title($selected_form_id)); ?></h2>
                    
                    <div style="margin-bottom: 20px;">
                        <?php submit_button('Instellingen Opslaan', 'primary', 'mfuc_save_settings', false, ['style' => 'float: right;']); ?>
                        <div style="clear: both;"></div>
                    </div>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="mfuc_enabled">Gebruikersaanmaak Inschakelen</label>
                            </th>
                            <td>
                                <input type="checkbox" name="mfuc_enabled" id="mfuc_enabled" value="1" <?php checked(isset($settings['enabled']) && $settings['enabled']); ?>>
                                <label for="mfuc_enabled">Automatisch WordPress gebruikers aanmaken voor dit formulier</label>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="mfuc_create_when">Wanneer Gebruiker Aanmaken</label>
                            </th>
                            <td>
                                <select name="mfuc_create_when" id="mfuc_create_when">
                                    <option value="after_payment" <?php selected(isset($settings['create_when']) ? $settings['create_when'] : 'after_payment', 'after_payment'); ?>>
                                        Alleen na succesvolle betaling
                                    </option>
                                    <option value="on_submit" <?php selected(isset($settings['create_when']) ? $settings['create_when'] : '', 'on_submit'); ?>>
                                        Direct bij formulier verzending
                                    </option>
                                    <option value="submit_activate_after_payment" <?php selected(isset($settings['create_when']) ? $settings['create_when'] : '', 'submit_activate_after_payment'); ?>>
                                        Bij verzending, maar alleen actief na betaling
                                    </option>
                                </select>
                                <p class="description">
                                    <strong>Na succesvolle betaling:</strong> Gebruiker wordt alleen aangemaakt nadat de betaling succesvol is.<br>
                                    <strong>Bij verzending:</strong> Gebruiker wordt direct aangemaakt wanneer het formulier wordt verzonden.<br>
                                    <strong>Bij verzending, actief na betaling:</strong> Gebruiker wordt aangemaakt bij verzending, maar krijgt pas toegang na betaling.
                                </p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="mfuc_user_role">Gebruikersrol</label>
                            </th>
                            <td>
                                <select name="mfuc_user_role" id="mfuc_user_role">
                                    <?php foreach ($roles as $role_key => $role_data): ?>
                                        <option value="<?php echo esc_attr($role_key); ?>" <?php selected(isset($settings['user_role']) ? $settings['user_role'] : 'subscriber', $role_key); ?>>
                                            <?php echo esc_html($role_data['name']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <p class="description">De WordPress rol die nieuwe gebruikers krijgen.</p>
                            </td>
                        </tr>
                    </table>
                    
                    <div style="margin-top: 20px;">
                        <?php submit_button('Instellingen Opslaan', 'primary', 'mfuc_save_settings', false, ['style' => 'float: right;']); ?>
                        <div style="clear: both;"></div>
                    </div>
                    
                    <hr>
                    
                    <h2>Bulk Import</h2>
                    <p>Maak WordPress gebruikers aan voor alle bestaande inschrijvingen van dit formulier die nog geen WordPress account hebben.</p>
                    <form method="post" action="<?php echo esc_url($form_action); ?>" style="display: inline-block;">
                        <?php wp_nonce_field('mfuc_bulk_import'); ?>
                        <input type="hidden" name="mfuc_form_id" value="<?php echo esc_attr($selected_form_id); ?>">
                        <?php submit_button('Bulk Import Uitvoeren', 'secondary', 'mfuc_bulk_import', false, [
                            'onclick' => "return confirm('Weet je zeker dat je WordPress gebruikers wilt aanmaken voor alle inschrijvingen van dit formulier? Dit kan even duren bij veel inschrijvingen.');"
                        ]); ?>
                    </form>
                    
                    <hr>
                    
                    <h2>Overzicht Aangemaakte Gebruikers</h2>
                    <?php mfuc_render_users_table($selected_form_id); ?>
                    
                <?php endif; ?>
            </form>
        <?php endif; ?>
    </div>
    
    <script>
    // Auto-submit form when form selection changes
    document.addEventListener('DOMContentLoaded', function() {
        var formSelect = document.getElementById('mfuc_form_id_select');
        if (formSelect) {
            formSelect.addEventListener('change', function() {
                var url = new URL(window.location.href);
                url.searchParams.set('form_id', this.value);
                window.location.href = url.toString();
            });
        }
    });
    </script>
    <?php
}

/**
 * Render users table for a specific form
 */
function mfuc_render_users_table($form_id) {
    global $wpdb;
    
    $table = $wpdb->prefix . 'mfuc_user_creation_log';
    
    // Get all user creations for this form
    $logs = $wpdb->get_results($wpdb->prepare(
        "SELECT * FROM $table WHERE form_id = %d ORDER BY created_at DESC LIMIT 50",
        $form_id
    ));
    
    if (empty($logs)) {
        echo '<p>Nog geen gebruikers aangemaakt voor dit formulier.</p>';
        return;
    }
    
    ?>
    <table class="wp-list-table widefat fixed striped">
        <thead>
            <tr>
                <th>Gebruiker</th>
                <th>E-mail</th>
                <th>Registratie ID</th>
                <th>Status</th>
                <th>Aangemaakt</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($logs as $log): ?>
                <?php
                $user = get_user_by('ID', $log->user_id);
                if (!$user) {
                    continue;
                }
                ?>
                <tr>
                    <td>
                        <strong><?php echo esc_html($user->display_name); ?></strong><br>
                        <small><?php echo esc_html($user->user_login); ?></small>
                    </td>
                    <td><?php echo esc_html($user->user_email); ?></td>
                    <td><?php echo esc_html($log->registration_id); ?></td>
                    <td>
                        <?php
                        $status_labels = [
                            'created' => 'Aangemaakt',
                            'updated' => 'Bijgewerkt',
                            'pending_activation' => 'Wachtend op betaling',
                            'activated' => 'Geactiveerd',
                            'error' => 'Fout',
                        ];
                        echo esc_html($status_labels[$log->status] ?? $log->status);
                        ?>
                    </td>
                    <td><?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($log->created_at))); ?></td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    <?php
}

