<?php
/*
Plugin Name: Folk Agenda
Plugin URI: https://folkagenda.nl/wordpress-plugin
Description: Fetch and display agenda data from mijn.folkagenda.nl.
Version: 1.0.1
Author: Folk Agenda
Author URI: https://folkagenda.nl
License: GPL2
*/

// Create the options page
function folkagenda_add_admin_menu() {
    add_options_page(
        'Folk Agenda',
        'Folk Agenda',
        'manage_options',
        'folkagenda',
        'folkagenda_options_page'
    );
}
add_action('admin_menu', 'folkagenda_add_admin_menu');

// Register settings and sections
function folkagenda_settings_init() {
    register_setting('folkagenda', 'folkagenda_settings');

    // Section: API Keys
    add_settings_section(
        'folkagenda_api_section',
        esc_html__('API Settings', 'folkagenda'),
        'folkagenda_api_section_callback',
        'folkagenda'
    );

    add_settings_field(
        'folkagenda_api_key',
        esc_html__('API Key', 'folkagenda'),
        'folkagenda_api_key_render',
        'folkagenda',
        'folkagenda_api_section'
    );

    add_settings_field(
        'folkagenda_personal_passkey',
        esc_html__('Personal Passkey', 'folkagenda'),
        'folkagenda_personal_passkey_render',
        'folkagenda',
        'folkagenda_api_section'
    );

    add_settings_field(
        'folkagenda_agenda_passkey',
        esc_html__('Agenda Passkey', 'folkagenda'),
        'folkagenda_agenda_passkey_render',
        'folkagenda',
        'folkagenda_api_section'
    );

    // Section: Descriptions
    add_settings_section(
        'folkagenda_description_section',
        esc_html__('Description Settings', 'folkagenda'),
        'folkagenda_description_section_callback',
        'folkagenda'
    );

    add_settings_field(
        'folkagenda_show_long_description',
        esc_html__('Show Long Description', 'folkagenda'),
        'folkagenda_show_long_description_render',
        'folkagenda',
        'folkagenda_description_section'
    );

    add_settings_field(
        'folkagenda_show_short_description',
        esc_html__('Show Short Description', 'folkagenda'),
        'folkagenda_show_short_description_render',
        'folkagenda',
        'folkagenda_description_section'
    );

    add_settings_field(
        'folkagenda_word_limit',
        esc_html__('Word Limit', 'folkagenda'),
        'folkagenda_word_limit_render',
        'folkagenda',
        'folkagenda_description_section'
    );

    // Section: Agenda
    add_settings_section(
        'folkagenda_agenda_section',
        esc_html__('Agenda Settings', 'folkagenda'),
        'folkagenda_agenda_section_callback',
        'folkagenda'
    );

    add_settings_field(
        'folkagenda_show_agenda_title',
        esc_html__('Show Agenda Title', 'folkagenda'),
        'folkagenda_show_agenda_title_render',
        'folkagenda',
        'folkagenda_agenda_section'
    );

    add_settings_field(
        'folkagenda_show_year',
        esc_html__('Show Year', 'folkagenda'),
        'folkagenda_show_year_render',
        'folkagenda',
        'folkagenda_agenda_section'
    );

    add_settings_field(
        'folkagenda_separator',
        esc_html__('Separator', 'folkagenda'),
        'folkagenda_separator_render',
        'folkagenda',
        'folkagenda_agenda_section'
    );

    // Section: Styles
    add_settings_section(
        'folkagenda_style_section',
        esc_html__('Styles', 'folkagenda'),
        'folkagenda_style_section_callback',
        'folkagenda'
    );

    add_settings_field(
        'folkagenda_custom_css',
        esc_html__('Custom CSS', 'folkagenda'),
        'folkagenda_custom_css_render',
        'folkagenda',
        'folkagenda_style_section'
    );
}
add_action('admin_init', 'folkagenda_settings_init');




// Section descriptions
function folkagenda_api_section_callback() {
    echo esc_html__('Enter your API keys to access the agenda data.', 'folkagenda');
}

function folkagenda_description_section_callback() {
    echo esc_html__('Set options for showing long and short descriptions, and word limit.', 'folkagenda');
}

function folkagenda_agenda_section_callback() {
    echo esc_html__('Configure the title, date display, and separator for agenda items.', 'folkagenda');
}

function folkagenda_style_section_callback() {
    echo esc_html__('You can override the default CSS styles here. To reset, deactivate and reactivate the plugin.', 'folkagenda');
}

// Rendering the input fields
function folkagenda_api_key_render() {
    $options = get_option('folkagenda_settings');
    ?>
    <input type='text' name='folkagenda_settings[folkagenda_api_key]' value='<?php echo esc_attr($options['folkagenda_api_key']); ?>'>
    <?php
}

function folkagenda_personal_passkey_render() {
    $options = get_option('folkagenda_settings');
    ?>
    <input type='text' name='folkagenda_settings[folkagenda_personal_passkey]' value='<?php echo esc_attr($options['folkagenda_personal_passkey']); ?>'>
    <?php
}

function folkagenda_agenda_passkey_render() {
    $options = get_option('folkagenda_settings');
    ?>
    <input type='text' name='folkagenda_settings[folkagenda_agenda_passkey]' value='<?php echo esc_attr($options['folkagenda_agenda_passkey']); ?>'>
    <?php
}

function folkagenda_show_long_description_render() {
    $options = get_option('folkagenda_settings');
    ?>
    <input type='checkbox' name='folkagenda_settings[folkagenda_show_long_description]' <?php checked($options['folkagenda_show_long_description'], 1); ?> value='1'>
    <?php
}

function folkagenda_show_short_description_render() {
    $options = get_option('folkagenda_settings');
    ?>
    <input type='checkbox' name='folkagenda_settings[folkagenda_show_short_description]' <?php checked($options['folkagenda_show_short_description'], 1); ?> value='1'>
    <?php
}

function folkagenda_word_limit_render() {
    $options = get_option('folkagenda_settings');
    ?>
    <input type='number' name='folkagenda_settings[folkagenda_word_limit]' value='<?php echo esc_attr($options['folkagenda_word_limit']); ?>'>
    <?php
}

function folkagenda_show_agenda_title_render() {
    $options = get_option('folkagenda_settings');
    ?>
    <input type='checkbox' name='folkagenda_settings[folkagenda_show_agenda_title]' <?php checked($options['folkagenda_show_agenda_title'], 1); ?> value='1'>
    <?php
}

function folkagenda_show_year_render() {
    $options = get_option('folkagenda_settings');
    ?>
    <input type='checkbox' name='folkagenda_settings[folkagenda_show_year]' <?php checked($options['folkagenda_show_year'], 1); ?> value='1'>
    <?php
}

function folkagenda_separator_render() {
    $options = get_option('folkagenda_settings');
    ?>
    <input type='text' name='folkagenda_settings[folkagenda_separator]' value='<?php echo esc_attr($options['folkagenda_separator']); ?>'>
    <?php
}

function folkagenda_custom_css_render() {
    $options = get_option('folkagenda_settings');
    ?>
    <textarea id="folkagenda_custom_css" name="folkagenda_settings[folkagenda_custom_css]" rows="10" cols="50"><?php echo esc_textarea($options['folkagenda_custom_css']); ?></textarea>
    <?php
}

// Options page HTML
function folkagenda_options_page() {
    ?>
    <form action='options.php' method='post'>
        <h1><?php echo esc_html__('Folk Agenda Settings', 'folkagenda'); ?></h1>
        <!-- General description goes here -->
        <p><?php echo esc_html__('Configure your Folk Agenda settings below. More info about what this all means can be found on https://folkagenda.nl/wordpress-plugin', 'folkagenda'); ?><br /></p>

        <?php
        settings_fields('folkagenda');
        do_settings_sections('folkagenda');
        submit_button();
        ?>
    </form>
    <?php
}

// Enqueue CodeMirror and initialize it for the Custom CSS textarea
function folkagenda_enqueue_code_editor() {
    // Enqueue WordPress CodeMirror editor
    $settings = wp_enqueue_code_editor(array('type' => 'text/css'));

    if ($settings) {
        wp_add_inline_script(
            'code-editor',
            sprintf('jQuery(function() { wp.codeEditor.initialize("folkagenda_custom_css", %s); });', wp_json_encode($settings))
        );
    }

    // Enqueue the CodeMirror editor styles and scripts
    wp_enqueue_script('wp-theme-plugin-editor');
    wp_enqueue_style('wp-codemirror');
}
add_action('admin_enqueue_scripts', 'folkagenda_enqueue_code_editor');


// Enqueue frontend CSS correctly for all pages where the shortcode is used
function folkagenda_enqueue_frontend_css() {
    // Only enqueue the CSS if the shortcode is found on the page
    if (has_shortcode(get_post()->post_content, 'folkagenda')) {
        wp_enqueue_style('folkagenda-frontend-css', plugin_dir_url(__FILE__) . 'css/folkagenda-frontend.css', array(), '1.0', 'all');
    }
}
add_action('wp_enqueue_scripts', 'folkagenda_enqueue_frontend_css');

// Enqueue custom CSS from the settings
function folkagenda_enqueue_custom_css() {
    $options = get_option('folkagenda_settings');
    if (!empty($options['folkagenda_custom_css'])) {
        // Print the custom CSS inline in the footer, so it overrides the default CSS
        echo '<style type="text/css">' . esc_html($options['folkagenda_custom_css']) . '</style>';
    }
}
add_action('wp_footer', 'folkagenda_enqueue_custom_css');

// Shortcode implementation
function folkagenda_shortcode() {



    $options = get_option('folkagenda_settings');
    $api_key = sanitize_text_field($options['folkagenda_api_key']);
    $personal_passkey = sanitize_text_field($options['folkagenda_personal_passkey']);
    $agenda_passkey = sanitize_text_field($options['folkagenda_agenda_passkey']);
    $show_year = isset($options['folkagenda_show_year']) ? (bool) $options['folkagenda_show_year'] : false;
    $show_shortdescription = isset($options['folkagenda_show_short_description']) ? (bool) $options['folkagenda_show_short_description'] : false;
    $show_longdescription = isset($options['folkagenda_show_long_description']) ? (bool) $options['folkagenda_show_long_description'] : false;
    $word_limit = isset($options['folkagenda_word_limit']) ? intval($options['folkagenda_word_limit']) : 20;
    $separator = isset($options['folkagenda_separator']) ? sanitize_text_field($options['folkagenda_separator']) : ' - ';
    $show_agenda_title = isset($options['folkagenda_show_agenda_title']) ? (bool) $options['folkagenda_show_agenda_title'] : false;

    // Set up the request URL and headers
    $url = "https://mijn.folkagenda.nl/api/$agenda_passkey/$personal_passkey";
    $args = array(
        'headers' => array(
            'X-API-Key' => $api_key
        )
    );

    // Fetch the data
    $response = wp_remote_get(esc_url_raw($url), $args);
	


    if (is_wp_error($response)) {
        return esc_html__('Unable to retrieve agenda data.', 'folkagenda');
    }

    $data = json_decode(wp_remote_retrieve_body($response), true);
    if (empty($data)) {
        return esc_html__('No agenda data found.', 'folkagenda');
    }

    // Begin output
    $output = "<div class='folkagenda-wrapper'>";

    foreach ($data as $item) {
        // Image
        $image = esc_url('https://mijn.folkagenda.nl/uploads/' . $item['image']);

        // Start and end times
        $starttime = gmdate("H:i", strtotime($item['start']));
        $endtime = isset($item['end']) && $item['end'] !== '0000-00-00 00:00:00' ? gmdate("H:i", strtotime($item['end'])) : null;
        $time = $endtime && $endtime != $starttime ? $starttime . ' - ' . $endtime : $starttime;

        // Date formatting
        $startdate = wp_date("d", strtotime($item['start']));
        $startmonth = wp_date("F", strtotime($item['start']));
        $enddate = wp_date("d", strtotime($item['end']));
        $endmonth = wp_date("F", strtotime($item['end']));

        // Construct dateline
        $dateline = $startdate;
        if ($startmonth !== $endmonth) {
            $dateline .= " $startmonth - $enddate $endmonth";
        } elseif ($startdate != $enddate) {
            $dateline .= " - $enddate $endmonth";
        } else {
            $dateline .= " $startmonth";
        }

        // Show year if enabled
        if ($show_year) {
            $dateline .= " " . wp_date("Y", strtotime($item['start']));
        }

        if($show_agenda_title){
            $dateline .= esc_html(" " . $separator . " " .  $item['agenda']);
        }




        // Title
        $title = esc_html($item['title']);

        // Description (short or long based on settings)
        $description = '';
        if ($show_shortdescription && isset($item['shortdescription'])) {
            $description = nl2br(esc_html(wp_trim_words($item['shortdescription'], $word_limit)));
        } elseif ($show_longdescription && isset($item['description'])) {
            $description = nl2br(esc_html(wp_trim_words($item['description'], $word_limit)));
        }

        // Links (Info and Tickets)
        $info_link = isset($item['info_link']) ? esc_url($item['info_link']) : null;
        $ticket_link = isset($item['ticket_link']) ? esc_url($item['ticket_link']) : null;

        $infobtn = $info_link ? "<a href='$info_link' target='_blank' class='folkagenda-button folkagenda-button-info'>" . esc_html__('Info', 'folkagenda') . "</a>" : '';
        $ticketbtn = $ticket_link ? "<a href='$ticket_link' target='_blank' class='folkagenda-button folkagenda-button-tickets'>" . esc_html__('Tickets', 'folkagenda') . "</a>" : '';

        // Location
        $location = esc_html($item['location']);


        // Build HTML for each event item
        $output .= "
        <div class='folkagenda-row'>
            <div class='folkagenda-1'>
                <div class='folkagenda-image-wrapper'>
                    <img class='folkagenda-img' src='$image' alt='" . esc_attr($title) . "' />
                </div>
            </div>
            <div class='folkagenda-2'>
                <div class='folkagenda-date'>$dateline</div>
                <div class='folkagenda-title'>$title</div>
                <div class='folkagenda-description'>$description</div>
                <div class='folkagenda-time'>$time</div>
                <div class='folkagenda-location'>$location</div>
            </div>
            <div class='folkagenda-3'>
                <div class='folkagenda-btn-div'>$ticketbtn</div>
                <div class='folkagenda-btn-div'>$infobtn</div>
            </div>
        </div>";
    }

    $output .= "</div>"; // Close the folkagenda container

    return $output;
}

// Register the shortcode after defining the function
add_shortcode('folkagenda', 'folkagenda_shortcode');



// Add a custom 'View Details' link on the plugins page
function folkagenda_plugin_action_links($links) {
    $details_link = '<a href="https://folkagenda.nl/wordpress-plugin" target="_blank">' . esc_html__('View Manual', 'folkagenda') . '</a>';
    array_unshift($links, $details_link);
    return $links;
}
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'folkagenda_plugin_action_links');

// Default CSS added when the plugin is activated
function folkagenda_activate() {
    $default_css = "
/* This class formats the display of the event date */
.folkagenda-date {}

/* This class formats the title of the event */
.folkagenda-title {}

/* This class formats the description of the event (either long or short, depending on settings) */
.folkagenda-description {}

/* This class wraps the image, enabling control over the image's size and proportions */
.folkagenda-image-wrapper {}

/* This class styles the event image itself (e.g., position, size, and fitting) */
.folkagenda-img {}

/* This class wraps each button, allowing for margin and positioning of buttons */
.folkagenda-btn-div {}

/* This class applies general styling to all buttons (e.g., padding, colors) */
.folkagenda-button {}

/* This class applies specific styling to the Tickets button */
.folkagenda-button-tickets {}

/* This class applies specific styling to the Info button */
.folkagenda-button-info {}

/* This class defines the layout for each event row in the grid system */
.folkagenda-row {}

/* This class aligns the buttons section in the event layout */
.folkagenda-3 {}

/* Responsive design for large screens, typically desktops */
@media screen and (min-width: 1200px) {
    /* Adjusts the layout for the entire row (displaying multiple columns) */
    .folkagenda-row {}

    /* Defines the position and size of the first section (image) */
    .folkagenda-1 {}

    /* Defines the position and size of the second section (date, title, description) */
    .folkagenda-2 {}

    /* Defines the position and size of the third section (buttons) */
    .folkagenda-3 {}
}

/* Responsive design for medium to smaller screens, such as tablets */
@media screen and (max-width: 1200px) {
    /* Adjusts the layout for the entire row */
    .folkagenda-row {}

    /* Full-width image on smaller screens */
    .folkagenda-1 {}

    /* Adjusts the title and description section */
    .folkagenda-2 {}

    /* Positions the button section in a separate row or adjusts its width */
    .folkagenda-3 {}
}

/* Responsive design for very small screens, such as mobile phones */
@media screen and (max-width: 400px) {
    /* Stack elements in a column layout for narrow screens */
    .folkagenda-row {}

    /* Full-width image for mobile screens */
    .folkagenda-1 {}

    /* Full-width title and description for mobile screens */
    .folkagenda-2 {}

    /* Full-width buttons, stacked below the other sections */
    .folkagenda-3 {}
}

    ";

    // Set default CSS if no CSS has been defined yet
    $options = get_option('folkagenda_settings');
    if (empty($options['folkagenda_custom_css'])) {
        $options['folkagenda_custom_css'] = $default_css;
        update_option('folkagenda_settings', $options);
    }
}

register_activation_hook(__FILE__, 'folkagenda_activate');

add_filter('pre_set_site_transient_update_plugins', function($transient) {
    if (empty($transient->checked)) return $transient;

    $remote = wp_remote_get('https://plugins.nietstezienhier.nl/folkagenda/update.json', [
        'timeout' => 15,
        'sslverify' => false,
    ]);
    if (is_wp_error($remote) || wp_remote_retrieve_response_code($remote) !== 200) return $transient;

    $plugin_data = json_decode(wp_remote_retrieve_body($remote));
    if (!$plugin_data) return $transient;

    $plugin_file = plugin_basename(__FILE__);
    $current_version = $transient->checked[$plugin_file] ?? null;

    if ($current_version && version_compare($plugin_data->version, $current_version, '>')) {
        $transient->response[$plugin_file] = (object)[
            'slug' => $plugin_data->slug,
            'new_version' => $plugin_data->version,
            'url' => $plugin_data->homepage,
            'package' => $plugin_data->download_url,
        ];
    }

    return $transient;
});

add_filter('plugins_api', function($result, $action, $args) {
    if ($action !== 'plugin_information' || $args->slug !== 'folkagenda') return $result;

    $remote = wp_remote_get('https://plugins.nietstezienhier.nl/folkagenda/update.json', [
        'timeout' => 15,
        'sslverify' => false,
    ]);
    if (is_wp_error($remote) || wp_remote_retrieve_response_code($remote) !== 200) return $result;

    $plugin_data = json_decode(wp_remote_retrieve_body($remote));
    if (!$plugin_data) return $result;

    return (object)[
        'name' => $plugin_data->name,
        'slug' => $plugin_data->slug,
        'version' => $plugin_data->version,
        'author' => $plugin_data->author,
        'homepage' => $plugin_data->homepage,
        'sections' => (array) $plugin_data->sections,
        'download_link' => $plugin_data->download_url,
    ];
}, 10, 3);