<?php
/**
 * Room Manager
 * Handles room/service detection and code management
 */

if (!defined('ABSPATH')) exit;

/**
 * Get all services/rooms from BookingPress
 * @return array List of services
 */
function bpac_get_bookingpress_services() {
    global $wpdb;
    
    $table = $wpdb->prefix . 'bookingpress_services';
    $services = $wpdb->get_results(
        "SELECT 
            bookingpress_service_id,
            bookingpress_service_name,
            bookingpress_category_id
        FROM $table
        ORDER BY bookingpress_service_position ASC, bookingpress_service_name ASC",
        ARRAY_A
    );
    
    return $services ? $services : [];
}

/**
 * Sync BookingPress services to our rooms table
 * Creates entries for new services, updates names for existing ones
 */
function bpac_sync_rooms_from_services() {
    global $wpdb;
    
    $services = bpac_get_bookingpress_services();
    $rooms_table = $wpdb->prefix . 'bpac_rooms';
    
    foreach ($services as $service) {
        $service_id = intval($service['bookingpress_service_id']);
        $service_name = sanitize_text_field($service['bookingpress_service_name']);
        $identifier = 'service_' . $service_id;
        
        // Check if room already exists
        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM $rooms_table WHERE room_identifier = %s",
            $identifier
        ));
        
        if ($existing) {
            // Update room name if it changed
            $wpdb->update(
                $rooms_table,
                ['room_name' => $service_name],
                ['room_identifier' => $identifier],
                ['%s'],
                ['%s']
            );
        } else {
            // Insert new room
            $wpdb->insert(
                $rooms_table,
                [
                    'room_identifier' => $identifier,
                    'room_name' => $service_name,
                    'access_code' => null
                ],
                ['%s', '%s', '%s']
            );
        }
    }
    
    // Remove rooms that no longer exist in BookingPress
    $all_identifiers = array_map(function($service) {
        return 'service_' . intval($service['bookingpress_service_id']);
    }, $services);
    
    if (!empty($all_identifiers)) {
        $placeholders = implode(',', array_fill(0, count($all_identifiers), '%s'));
        $query = "DELETE FROM $rooms_table WHERE room_identifier NOT IN ($placeholders)";
        $wpdb->query($wpdb->prepare($query, $all_identifiers));
    }
}

/**
 * Get all rooms with their codes
 * @return array List of rooms
 */
function bpac_get_all_rooms() {
    global $wpdb;
    
    $table = $wpdb->prefix . 'bpac_rooms';
    
    // First get raw values to see what's actually in the database
    $raw_rooms = $wpdb->get_results(
        "SELECT id, room_identifier, room_name, access_code, last_updated 
         FROM $table 
         ORDER BY room_name ASC",
        ARRAY_A
    );
    
    error_log("BPAC DEBUG: Retrieved " . count($raw_rooms ? $raw_rooms : []) . " rooms from database");
    
    // Process rooms
    $rooms = [];
    if ($raw_rooms) {
        foreach ($raw_rooms as $raw_room) {
            // Get raw value from database (can be NULL)
            $raw_code = isset($raw_room['access_code']) ? $raw_room['access_code'] : null;
            
            // Log the raw value to see what's actually in DB
            error_log("BPAC DEBUG: Room {$raw_room['id']} ({$raw_room['room_name']}): RAW access_code = " . var_export($raw_code, true) . " (type: " . gettype($raw_code) . ")");
            
            // Convert NULL to empty string, otherwise use the value
            $processed_code = ($raw_code === null) ? '' : trim((string) $raw_code);
            
            // Build room array with processed code
            $room = $raw_room;
            $room['access_code'] = $processed_code;
            
            error_log("BPAC DEBUG: Room {$raw_room['id']} ({$raw_room['room_name']}): PROCESSED access_code = " . var_export($processed_code, true));
            
            $rooms[] = $room;
        }
    }
    
    return $rooms;
}

/**
 * Get room by identifier
 * @param string $identifier Room identifier
 * @return object|null Room data or null
 */
function bpac_get_room_by_identifier($identifier) {
    global $wpdb;
    
    $table = $wpdb->prefix . 'bpac_rooms';
    $room = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM $table WHERE room_identifier = %s",
        $identifier
    ));
    
    return $room;
}

/**
 * Update room access code
 * @param int $room_id Room ID
 * @param string $access_code Access code (can be empty string to clear)
 * @return bool Success
 */
function bpac_update_room_code($room_id, $access_code) {
    global $wpdb;
    
    $table = $wpdb->prefix . 'bpac_rooms';
    
    // Trim and sanitize - keep as string even if empty
    $access_code = trim(sanitize_text_field($access_code));
    $room_id = intval($room_id);
    
    error_log("BPAC DEBUG: bpac_update_room_code called with room_id=$room_id, access_code='$access_code'");
    
    // First, verify the room exists
    $room_exists = $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(*) FROM $table WHERE id = %d",
        $room_id
    ));
    
    error_log("BPAC DEBUG: Room exists check: " . var_export($room_exists, true));
    
    if (!$room_exists) {
        error_log("BPAC DEBUG: Room does not exist!");
        return false;
    }
    
    // Get current value before update
    $current_value = $wpdb->get_var($wpdb->prepare(
        "SELECT access_code FROM $table WHERE id = %d",
        $room_id
    ));
    error_log("BPAC DEBUG: Current value in DB: " . var_export($current_value, true));
    
    // Use direct SQL UPDATE to ensure it works
    // Store empty string as empty string (not NULL)
    $sql = $wpdb->prepare(
        "UPDATE $table SET access_code = %s WHERE id = %d",
        $access_code,
        $room_id
    );
    
    error_log("BPAC DEBUG: SQL query: " . $sql);
    
    $result = $wpdb->query($sql);
    
    error_log("BPAC DEBUG: Query result: " . var_export($result, true));
    error_log("BPAC DEBUG: Last error: " . $wpdb->last_error);
    
    // Check if query executed successfully
    if ($result === false) {
        error_log("BPAC DEBUG: Query failed!");
        return false;
    }
    
    // Always verify by reading back the value immediately
    $saved_value = $wpdb->get_var($wpdb->prepare(
        "SELECT access_code FROM $table WHERE id = %d",
        $room_id
    ));
    
    error_log("BPAC DEBUG: Value after update: " . var_export($saved_value, true));
    
    // Normalize both values for comparison
    // Handle both NULL and empty string as empty
    $saved_normalized = ($saved_value !== null && $saved_value !== '') ? trim($saved_value) : '';
    $expected_normalized = trim($access_code);
    
    error_log("BPAC DEBUG: Saved normalized: '$saved_normalized', Expected normalized: '$expected_normalized'");
    
    // Return true if the value matches what we tried to save
    $success = ($saved_normalized === $expected_normalized);
    
    error_log("BPAC DEBUG: Success: " . var_export($success, true));
    
    return $success;
}

/**
 * Get access code for a service ID
 * @param int $service_id BookingPress service ID
 * @return string|null Access code or null
 */
function bpac_get_access_code_for_service($service_id) {
    $identifier = 'service_' . intval($service_id);
    $room = bpac_get_room_by_identifier($identifier);
    
    return $room && !empty($room->access_code) ? $room->access_code : null;
}




