<?php
/**
 * Email Handler
 * Handles sending access codes to customers
 */

if (!defined('ABSPATH')) exit;

/**
 * Get bookings that need access codes sent today
 * @return array List of bookings
 */
function bpac_get_bookings_for_today() {
    global $wpdb;
    
    $table = $wpdb->prefix . 'bookingpress_appointment_bookings';
    
    // Check if we should send day before
    $send_day_before = get_option('bpac_send_day_before', 0);
    
    if ($send_day_before) {
        // Send for bookings tomorrow (so they receive it today, the day before)
        $query = $wpdb->prepare(
            "SELECT 
                b.bookingpress_appointment_booking_id,
                b.bookingpress_customer_name,
                b.bookingpress_customer_email,
                b.bookingpress_appointment_date,
                b.bookingpress_appointment_time,
                b.bookingpress_service_id,
                b.bookingpress_service_name,
                b.bookingpress_paid_amount,
                b.bookingpress_due_amount,
                b.bookingpress_appointment_status
            FROM $table b
            WHERE b.bookingpress_paid_amount > 0
                AND b.bookingpress_appointment_status IN (1, 2)
                AND b.bookingpress_appointment_date = DATE_ADD(CURDATE(), INTERVAL 1 DAY)
            ORDER BY b.bookingpress_appointment_date ASC, b.bookingpress_appointment_time ASC",
            []
        );
    } else {
        // Send for bookings today and tomorrow (original behavior)
        $query = $wpdb->prepare(
            "SELECT 
                b.bookingpress_appointment_booking_id,
                b.bookingpress_customer_name,
                b.bookingpress_customer_email,
                b.bookingpress_appointment_date,
                b.bookingpress_appointment_time,
                b.bookingpress_service_id,
                b.bookingpress_service_name,
                b.bookingpress_paid_amount,
                b.bookingpress_due_amount,
                b.bookingpress_appointment_status
            FROM $table b
            WHERE b.bookingpress_paid_amount > 0
                AND b.bookingpress_appointment_status IN (1, 2)
                AND b.bookingpress_appointment_date >= CURDATE()
                AND b.bookingpress_appointment_date <= DATE_ADD(CURDATE(), INTERVAL 1 DAY)
            ORDER BY b.bookingpress_appointment_date ASC, b.bookingpress_appointment_time ASC",
            []
        );
    }
    
    $bookings = $wpdb->get_results($query, ARRAY_A);
    
    return $bookings ? $bookings : [];
}

/**
 * Check if email was already sent for this booking today
 * @param int $booking_id Booking ID
 * @return bool True if already sent
 */
function bpac_email_already_sent_today($booking_id) {
    global $wpdb;
    
    $table = $wpdb->prefix . 'bpac_email_log';
    $today = date('Y-m-d');
    
    $count = $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(*) FROM $table 
        WHERE booking_id = %d 
        AND DATE(sent_at) = %s 
        AND status = 'sent'",
        $booking_id,
        $today
    ));
    
    return $count > 0;
}

/**
 * Log email sent
 * @param int $booking_id Booking ID
 * @param int $room_id Room ID
 * @param string $email Customer email
 * @param string $status Status (sent/failed)
 */
function bpac_log_email($booking_id, $room_id, $email, $status = 'sent') {
    global $wpdb;
    
    $table = $wpdb->prefix . 'bpac_email_log';
    
    $wpdb->insert(
        $table,
        [
            'booking_id' => intval($booking_id),
            'room_id' => intval($room_id),
            'customer_email' => sanitize_email($email),
            'status' => sanitize_text_field($status)
        ],
        ['%d', '%d', '%s', '%s']
    );
}

/**
 * Send access code email to customer
 * @param array $booking Booking data
 * @param string $access_code Access code
 * @return bool Success
 */
function bpac_send_access_code_email($booking, $access_code) {
    $email = sanitize_email($booking['bookingpress_customer_email']);
    $name = sanitize_text_field($booking['bookingpress_customer_name']);
    $service_name = sanitize_text_field($booking['bookingpress_service_name']);
    $date = sanitize_text_field($booking['bookingpress_appointment_date']);
    $time = sanitize_text_field($booking['bookingpress_appointment_time']);
    
    if (empty($email)) {
        return false;
    }
    
    // Format date
    $date_obj = DateTime::createFromFormat('Y-m-d', $date);
    $formatted_date = $date_obj ? $date_obj->format('d-m-Y') : $date;
    
    // Format time
    $time_obj = DateTime::createFromFormat('H:i:s', $time);
    $formatted_time = $time_obj ? $time_obj->format('H:i') : $time;
    
    // Get email template from settings or use default
    $email_subject_template = get_option('bpac_email_subject', 'Uw toegangscode voor {ruimte}');
    $email_template = get_option('bpac_email_template', "Beste {naam},\n\nHier is uw toegangscode voor de ruimte die u heeft geboekt:\n\nRuimte: {ruimte}\nDatum: {datum}\nTijd: {tijd}\nToegangscode: {code}\n\nTot binnenkort!");
    
    // Replace placeholders in subject
    $subject = str_replace(
        ['{naam}', '{ruimte}', '{datum}', '{tijd}', '{code}'],
        [$name, $service_name, $formatted_date, $formatted_time, $access_code],
        $email_subject_template
    );
    
    // Replace placeholders in message
    $message = str_replace(
        ['{naam}', '{ruimte}', '{datum}', '{tijd}', '{code}'],
        [$name, $service_name, $formatted_date, $formatted_time, $access_code],
        $email_template
    );
    
    $headers = ['Content-Type: text/plain; charset=UTF-8'];
    
    $sent = wp_mail($email, $subject, $message, $headers);
    
    return $sent;
}

/**
 * Send access codes for all bookings today
 */
function bpac_send_access_codes_for_today() {
    // Check if auto send is enabled
    $enable_auto_send = get_option('bpac_enable_auto_send', 1);
    if (!$enable_auto_send) {
        error_log('BPAC: Automatische verzending is uitgeschakeld');
        return ['sent' => 0, 'failed' => 0, 'skipped' => true];
    }
    
    $bookings = bpac_get_bookings_for_today();
    $sent_count = 0;
    $failed_count = 0;
    
    foreach ($bookings as $booking) {
        $booking_id = intval($booking['bookingpress_appointment_booking_id']);
        $service_id = intval($booking['bookingpress_service_id']);
        
        // Skip if already sent today
        if (bpac_email_already_sent_today($booking_id)) {
            continue;
        }
        
        // Get access code for this service
        $access_code = bpac_get_access_code_for_service($service_id);
        
        if (empty($access_code)) {
            // No access code set for this room, skip
            continue;
        }
        
        // Get room ID for logging
        $identifier = 'service_' . $service_id;
        $room = bpac_get_room_by_identifier($identifier);
        $room_id = $room ? $room->id : null;
        
        // Send email
        $sent = bpac_send_access_code_email($booking, $access_code);
        
        // Log result
        bpac_log_email(
            $booking_id,
            $room_id,
            $booking['bookingpress_customer_email'],
            $sent ? 'sent' : 'failed'
        );
        
        if ($sent) {
            $sent_count++;
        } else {
            $failed_count++;
        }
    }
    
    // Log summary (optional - could be stored in options)
    error_log(sprintf(
        'BPAC: Sent %d access code emails, %d failed',
        $sent_count,
        $failed_count
    ));
    
    return ['sent' => $sent_count, 'failed' => $failed_count];
}

/**
 * Send access codes immediately (manual trigger, ignores auto_send setting)
 * @param bool $ignore_already_sent If true, will resend even if already sent today
 * @return array Result with sent/failed counts
 */
function bpac_send_access_codes_now($ignore_already_sent = false) {
    $bookings = bpac_get_bookings_for_today();
    $sent_count = 0;
    $failed_count = 0;
    $skipped_count = 0;
    
    foreach ($bookings as $booking) {
        $booking_id = intval($booking['bookingpress_appointment_booking_id']);
        $service_id = intval($booking['bookingpress_service_id']);
        
        // Skip if already sent today (unless we're ignoring that)
        if (!$ignore_already_sent && bpac_email_already_sent_today($booking_id)) {
            $skipped_count++;
            continue;
        }
        
        // Get access code for this service
        $access_code = bpac_get_access_code_for_service($service_id);
        
        if (empty($access_code)) {
            // No access code set for this room, skip
            $skipped_count++;
            continue;
        }
        
        // Get room ID for logging
        $identifier = 'service_' . $service_id;
        $room = bpac_get_room_by_identifier($identifier);
        $room_id = $room ? $room->id : null;
        
        // Send email
        $sent = bpac_send_access_code_email($booking, $access_code);
        
        // Log result
        bpac_log_email(
            $booking_id,
            $room_id,
            $booking['bookingpress_customer_email'],
            $sent ? 'sent' : 'failed'
        );
        
        if ($sent) {
            $sent_count++;
        } else {
            $failed_count++;
        }
    }
    
    // Log summary
    error_log(sprintf(
        'BPAC (Manual): Sent %d access code emails, %d failed, %d skipped',
        $sent_count,
        $failed_count,
        $skipped_count
    ));
    
    return [
        'sent' => $sent_count, 
        'failed' => $failed_count,
        'skipped' => $skipped_count,
        'total_bookings' => count($bookings)
    ];
}

