<?php
/**
 * Database Schema Check System
 * Checks if BookingPress database structure is compatible
 */

if (!defined('ABSPATH')) exit;

/**
 * Check BookingPress database schema compatibility
 * @return array Status information
 */
function bpac_check_bookingpress_schema() {
    global $wpdb;
    
    $results = [
        'status' => 'ok',
        'messages' => [],
        'missing_columns' => [],
    ];
    
    // Check appointment_bookings table
    $table_bookings = $wpdb->prefix . 'bookingpress_appointment_bookings';
    $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_bookings));
    
    if (!$table_exists) {
        $results['status'] = 'error';
        $results['messages'][] = 'BookingPress appointment_bookings tabel niet gevonden. Zorg ervoor dat BookingPress is geïnstalleerd en geactiveerd.';
        return $results;
    }
    
    // Check required columns for appointment_bookings
    $expected_columns_bookings = [
        'bookingpress_appointment_booking_id',
        'bookingpress_customer_email',
        'bookingpress_customer_name',
        'bookingpress_appointment_date',
        'bookingpress_appointment_time',
        'bookingpress_service_id',
        'bookingpress_service_name',
        'bookingpress_paid_amount',
        'bookingpress_appointment_status'
    ];
    
    $actual_columns = $wpdb->get_col("SHOW COLUMNS FROM $table_bookings");
    $missing = array_diff($expected_columns_bookings, $actual_columns);
    
    if (!empty($missing)) {
        $results['status'] = 'warning';
        $results['messages'][] = 'BookingPress database structuur is gewijzigd. Sommige verwachte kolommen ontbreken.';
        $results['missing_columns'] = $missing;
    }
    
    // Check services table
    $table_services = $wpdb->prefix . 'bookingpress_services';
    $services_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_services));
    
    if (!$services_exists) {
        $results['status'] = 'error';
        $results['messages'][] = 'BookingPress services tabel niet gevonden.';
        return $results;
    }
    
    // Check required columns for services
    $expected_columns_services = [
        'bookingpress_service_id',
        'bookingpress_service_name'
    ];
    
    $actual_columns_services = $wpdb->get_col("SHOW COLUMNS FROM $table_services");
    $missing_services = array_diff($expected_columns_services, $actual_columns_services);
    
    if (!empty($missing_services)) {
        $results['status'] = 'warning';
        $results['messages'][] = 'BookingPress services tabel mist verwachte kolommen.';
        $results['missing_columns'] = array_merge($results['missing_columns'], $missing_services);
    }
    
    if ($results['status'] === 'ok') {
        $results['messages'][] = 'BookingPress database structuur is compatibel.';
    }
    
    return $results;
}

/**
 * Display admin notice if schema check fails
 */
add_action('admin_notices', 'bpac_schema_check_notice');
function bpac_schema_check_notice() {
    if (!current_user_can('manage_options')) return;
    
    $schema_check = bpac_check_bookingpress_schema();
    
    if ($schema_check['status'] === 'ok') {
        return;
    }
    
    $class = $schema_check['status'] === 'error' ? 'error' : 'warning';
    $message = implode(' ', $schema_check['messages']);
    
    if (!empty($schema_check['missing_columns'])) {
        $message .= ' Ontbrekende kolommen: ' . implode(', ', $schema_check['missing_columns']);
    }
    
    echo '<div class="notice notice-' . esc_attr($class) . ' is-dismissible">';
    echo '<p><strong>BookingPress Access Codes:</strong> ' . esc_html($message) . '</p>';
    echo '</div>';
}




