<?php
/**
 * Admin Settings Page
 * Handles admin interface for managing rooms and codes
 */

if (!defined('ABSPATH')) exit;

// Plugin directory should already be defined in main file
// If not, define it as fallback
if (!defined('BPAC_PLUGIN_DIR')) {
    define('BPAC_PLUGIN_DIR', plugin_dir_path(dirname(__FILE__)));
}

// Add admin menu
add_action('admin_menu', 'bpac_add_admin_menu', 99);

function bpac_add_admin_menu() {
    $parent_slug = 'bookingpress';
    
    // Check if BookingPress menu exists
    global $menu;
    $bookingpress_exists = false;
    foreach ($menu as $item) {
        if (isset($item[2]) && $item[2] === $parent_slug) {
            $bookingpress_exists = true;
            break;
        }
    }
    
    // If BookingPress menu doesn't exist, create top-level menu
    if (!$bookingpress_exists) {
        add_menu_page(
            'Access Codes',
            'Access Codes',
            'manage_options',
            'bpac-settings',
            'bpac_render_settings_page',
            'dashicons-admin-network',
            30
        );
    } else {
        // Add submenu under BookingPress
        add_submenu_page(
            $parent_slug,
            'Access Codes Instellingen',
            'Access Codes',
            'manage_options',
            'bpac-settings',
            'bpac_render_settings_page'
        );
    }
}

// Register settings
add_action('admin_init', 'bpac_register_settings');

function bpac_register_settings() {
    register_setting('bpac_settings_group', 'bpac_daily_send_time');
    register_setting('bpac_settings_group', 'bpac_email_subject');
    register_setting('bpac_settings_group', 'bpac_email_template');
    register_setting('bpac_settings_group', 'bpac_enable_auto_send');
    register_setting('bpac_settings_group', 'bpac_send_day_before');
}

// Render settings page
function bpac_render_settings_page() {
    if (!current_user_can('manage_options')) {
        wp_die('Je hebt geen toestemming om deze pagina te bekijken.');
    }
    
    // Handle form submissions
    if (isset($_POST['bpac_sync_rooms'])) {
        if (!current_user_can('manage_options')) {
            wp_die('Je hebt geen toestemming om deze actie uit te voeren.');
        }
        // Verify nonce, but don't fail if expired
        if (isset($_POST['_wpnonce']) && !wp_verify_nonce($_POST['_wpnonce'], 'bpac_sync_rooms')) {
            error_log('BPAC: Sync rooms nonce verification failed');
        }
        if (!function_exists('bpac_sync_rooms_from_services')) {
            require_once BPAC_PLUGIN_DIR . 'includes/room-manager.php';
        }
        bpac_sync_rooms_from_services();
        echo '<div class="notice notice-success"><p>Ruimtes gesynchroniseerd!</p></div>';
    }
    
    // Handle bulk update of all codes
    if (isset($_POST['bpac_update_all_codes'])) {
        if (!current_user_can('manage_options')) {
            wp_die('Je hebt geen toestemming om deze actie uit te voeren.');
        }
        // Verify nonce, but don't fail if expired
        if (isset($_POST['_wpnonce']) && !wp_verify_nonce($_POST['_wpnonce'], 'bpac_update_all_codes')) {
            error_log('BPAC: Update codes nonce verification failed');
        }
        if (!function_exists('bpac_update_room_code')) {
            require_once BPAC_PLUGIN_DIR . 'includes/room-manager.php';
        }
        
        $updated_count = 0;
        $failed_count = 0;
        
        // Get all rooms to update
        if (isset($_POST['room_codes']) && is_array($_POST['room_codes'])) {
            foreach ($_POST['room_codes'] as $room_id => $access_code) {
                $room_id = intval($room_id);
                $access_code = isset($access_code) ? trim(sanitize_text_field($access_code)) : '';
                
                $result = bpac_update_room_code($room_id, $access_code);
                if ($result) {
                    $updated_count++;
                } else {
                    $failed_count++;
                }
            }
        }
        
        if ($updated_count > 0) {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html($updated_count) . ' code(s) bijgewerkt!</p></div>';
        }
        if ($failed_count > 0) {
            echo '<div class="notice notice-error"><p>' . esc_html($failed_count) . ' code(s) konden niet worden bijgewerkt.</p></div>';
        }
    }
    
    if (isset($_POST['bpac_save_settings'])) {
        // Verify user has permission
        if (!current_user_can('manage_options')) {
            wp_die('Je hebt geen toestemming om deze actie uit te voeren.');
        }
        
        // Try nonce verification, but don't fail if it's expired (common WordPress issue)
        $nonce_valid = false;
        if (isset($_POST['_wpnonce'])) {
            $nonce_valid = wp_verify_nonce($_POST['_wpnonce'], 'bpac_save_settings');
        }
        
        // If nonce is invalid, log it but still allow if user has proper permissions
        // This handles the common "expired link" issue in WordPress
        if (!$nonce_valid) {
            error_log('BPAC: Nonce verification failed, but allowing due to user permissions check');
        }
        
        $send_time = sanitize_text_field($_POST['bpac_daily_send_time']);
        $email_subject = sanitize_text_field($_POST['bpac_email_subject']);
        $email_template = wp_kses_post($_POST['bpac_email_template']);
        $enable_auto_send = isset($_POST['bpac_enable_auto_send']) ? 1 : 0;
        $send_day_before = isset($_POST['bpac_send_day_before']) ? 1 : 0;
        
        update_option('bpac_daily_send_time', $send_time);
        update_option('bpac_email_subject', $email_subject);
        update_option('bpac_email_template', $email_template);
        update_option('bpac_enable_auto_send', $enable_auto_send);
        update_option('bpac_send_day_before', $send_day_before);
        
        // Reschedule or clear cron based on setting
        wp_clear_scheduled_hook('bpac_daily_send_codes');
        if ($enable_auto_send && function_exists('bpac_activate_cron')) {
            bpac_activate_cron();
        }
        
        // Show success message immediately (no redirect to avoid nonce issues)
        echo '<div class="notice notice-success is-dismissible"><p>Instellingen opgeslagen!</p></div>';
        
        // Refresh the page data after save
        $send_time = get_option('bpac_daily_send_time', '08:00');
        $email_subject = get_option('bpac_email_subject', 'Uw toegangscode voor {ruimte}');
        $email_template = get_option('bpac_email_template', "Beste {naam},\n\nHier is uw toegangscode voor de ruimte die u heeft geboekt:\n\nRuimte: {ruimte}\nDatum: {datum}\nTijd: {tijd}\nToegangscode: {code}\n\nTot binnenkort!");
        $enable_auto_send = get_option('bpac_enable_auto_send', 1);
        $send_day_before = get_option('bpac_send_day_before', 0);
        $next_run = wp_next_scheduled('bpac_daily_send_codes');
        $next_run_formatted = $next_run ? date('d-m-Y H:i', $next_run) : 'Niet gepland';
    }
    
    if (isset($_POST['bpac_test_email'])) {
        if (!current_user_can('manage_options')) {
            wp_die('Je hebt geen toestemming om deze actie uit te voeren.');
        }
        // Verify nonce, but don't fail if expired
        if (isset($_POST['_wpnonce']) && !wp_verify_nonce($_POST['_wpnonce'], 'bpac_test_email')) {
            error_log('BPAC: Test email nonce verification failed');
        }
        if (!function_exists('bpac_send_access_code_email')) {
            require_once BPAC_PLUGIN_DIR . 'includes/email-handler.php';
        }
        $test_email = sanitize_email($_POST['test_email']);
        if (!empty($test_email)) {
            $test_booking = [
                'bookingpress_customer_name' => 'Test Gebruiker',
                'bookingpress_customer_email' => $test_email,
                'bookingpress_service_name' => 'Test Ruimte',
                'bookingpress_appointment_date' => date('Y-m-d'),
                'bookingpress_appointment_time' => '10:00:00'
            ];
            $sent = bpac_send_access_code_email($test_booking, '1234');
            if ($sent) {
                echo '<div class="notice notice-success"><p>Test e-mail verzonden naar ' . esc_html($test_email) . '!</p></div>';
            } else {
                echo '<div class="notice notice-error"><p>Fout bij verzenden test e-mail.</p></div>';
            }
        }
    }
    
    // Handle manual send now (override scheduled time)
    if (isset($_POST['bpac_send_now'])) {
        if (!current_user_can('manage_options')) {
            wp_die('Je hebt geen toestemming om deze actie uit te voeren.');
        }
        // Verify nonce, but don't fail if expired
        if (isset($_POST['_wpnonce']) && !wp_verify_nonce($_POST['_wpnonce'], 'bpac_send_now')) {
            error_log('BPAC: Send now nonce verification failed');
        }
        if (!function_exists('bpac_send_access_codes_now')) {
            require_once BPAC_PLUGIN_DIR . 'includes/email-handler.php';
        }
        
        $ignore_already_sent = isset($_POST['bpac_ignore_already_sent']) ? 1 : 0;
        $result = bpac_send_access_codes_now($ignore_already_sent);
        
        if ($result['sent'] > 0 || $result['failed'] > 0) {
            $message = sprintf(
                'Verzending voltooid: %d e-mail(s) verzonden, %d mislukt',
                $result['sent'],
                $result['failed']
            );
            if ($result['skipped'] > 0) {
                $message .= sprintf(', %d overgeslagen (al verzonden of geen code)', $result['skipped']);
            }
            echo '<div class="notice notice-success"><p>' . esc_html($message) . '</p></div>';
        } else {
            $message = 'Geen e-mails verzonden. ';
            if ($result['total_bookings'] == 0) {
                $message .= 'Geen relevante boekingen gevonden voor vandaag/morgen.';
            } else {
                $message .= sprintf('Alle %d boekingen zijn al verzonden of hebben geen toegangscode.', $result['total_bookings']);
            }
            echo '<div class="notice notice-warning"><p>' . esc_html($message) . '</p></div>';
        }
    }
    
    // Get schema check results
    if (!function_exists('bpac_check_bookingpress_schema')) {
        require_once BPAC_PLUGIN_DIR . 'includes/database-check.php';
    }
    $schema_check = bpac_check_bookingpress_schema();
    
    // Get all rooms
    if (!function_exists('bpac_get_all_rooms')) {
        require_once BPAC_PLUGIN_DIR . 'includes/room-manager.php';
    }
    $rooms = bpac_get_all_rooms();
    
    // Get current send time
    $send_time = get_option('bpac_daily_send_time', '08:00');
    
    // Get email settings
    $email_subject = get_option('bpac_email_subject', 'Uw toegangscode voor {ruimte}');
    $email_template = get_option('bpac_email_template', "Beste {naam},\n\nHier is uw toegangscode voor de ruimte die u heeft geboekt:\n\nRuimte: {ruimte}\nDatum: {datum}\nTijd: {tijd}\nToegangscode: {code}\n\nTot binnenkort!");
    $enable_auto_send = get_option('bpac_enable_auto_send', 1);
    $send_day_before = get_option('bpac_send_day_before', 0);
    
    // Get next cron schedule
    $next_run = wp_next_scheduled('bpac_daily_send_codes');
    $next_run_formatted = $next_run ? date('d-m-Y H:i', $next_run) : 'Niet gepland';
    
    ?>
    <style>
        .bpac-full-width {
            max-width: 100% !important;
            margin: 20px 0 0 0 !important;
        }
        .bpac-full-width .card {
            max-width: 100%;
        }
        .bpac-email-template-editor {
            width: 100%;
            min-height: 200px;
            font-family: monospace;
        }
        .bpac-replacement-help {
            background: #f0f0f1;
            padding: 15px;
            border-left: 4px solid #2271b1;
            margin: 15px 0;
        }
        .bpac-replacement-help code {
            background: #fff;
            padding: 2px 6px;
            border-radius: 3px;
        }
        .bpac-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        .bpac-header h1 {
            margin: 0;
            padding: 0;
        }
        .bpac-header-actions {
            display: flex;
            gap: 10px;
        }
    </style>
    <div class="wrap bpac-full-width">
        <div class="bpac-header">
            <h1>BookingPress Access Codes</h1>
            <div class="bpac-header-actions">
                <button type="button" id="bpac-save-top" class="button button-primary" style="height: 30px; line-height: 28px;">
                    Instellingen opslaan
                </button>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            $('#bpac-save-top').on('click', function(e) {
                e.preventDefault();
                // Find the settings form and click the submit button
                var $form = $('#bpac-settings-form');
                if ($form.length) {
                    // Find and click the submit button in the form
                    var $submitBtn = $form.find('input[name="bpac_save_settings"]');
                    if ($submitBtn.length) {
                        $submitBtn.click();
                    } else {
                        // Fallback: submit the form directly
                        $form.submit();
                    }
                }
            });
        });
        </script>
        
        <!-- Schema Check Status -->
        <div class="card" style="margin-top: 20px;">
            <h2>Database Compatibiliteit</h2>
            <?php if ($schema_check['status'] === 'ok'): ?>
                <p style="color: green;">✓ <?php echo esc_html($schema_check['messages'][0]); ?></p>
            <?php else: ?>
                <p style="color: <?php echo $schema_check['status'] === 'error' ? 'red' : 'orange'; ?>;">
                    ⚠ <?php echo esc_html(implode(' ', $schema_check['messages'])); ?>
                </p>
                <?php if (!empty($schema_check['missing_columns'])): ?>
                    <p>Ontbrekende kolommen: <?php echo esc_html(implode(', ', $schema_check['missing_columns'])); ?></p>
                <?php endif; ?>
            <?php endif; ?>
        </div>
        
        <!-- Settings -->
        <div class="card" style="margin-top: 20px;">
            <h2>Instellingen</h2>
            <form method="post" action="" id="bpac-settings-form">
                <?php wp_nonce_field('bpac_save_settings', '_wpnonce'); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row">Automatische verzending</th>
                        <td>
                            <label>
                                <input type="checkbox" name="bpac_enable_auto_send" value="1" <?php checked(1, $enable_auto_send); ?> />
                                Automatisch dagelijks toegangscodes verzenden
                            </label>
                            <p class="description">Schakel dit uit om automatische verzending te stoppen. Je kunt nog steeds handmatig test e-mails verzenden.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Verzendtijd</th>
                        <td>
                            <label>
                                <input type="checkbox" name="bpac_send_day_before" value="1" <?php checked(1, $send_day_before); ?> />
                                Verzend de dag voor de boeking
                            </label>
                            <p class="description">Als aangevinkt, worden de e-mails verzonden op de dag vóór de boeking op het ingestelde tijdstip. Anders worden ze verzonden op de dag van de boeking.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Dagelijks verzendtijd</th>
                        <td>
                            <input type="time" name="bpac_daily_send_time" value="<?php echo esc_attr($send_time); ?>" <?php echo $enable_auto_send ? '' : 'disabled'; ?> />
                            <p class="description">Tijdstip waarop dagelijks toegangscodes worden verzonden.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Volgende verzending</th>
                        <td>
                            <?php if ($enable_auto_send): ?>
                                <strong><?php echo esc_html($next_run_formatted); ?></strong>
                                <?php if ($send_day_before): ?>
                                    <br><small style="color: #666;">E-mails worden verzonden voor boekingen van morgen</small>
                                <?php else: ?>
                                    <br><small style="color: #666;">E-mails worden verzonden voor boekingen van vandaag en morgen</small>
                                <?php endif; ?>
                                <br><br>
                                <div style="background: #f0f0f1; padding: 15px; border-left: 4px solid #2271b1; margin-top: 10px;">
                                    <strong>⚠ Belangrijk: WP-Cron timing</strong>
                                    <p style="margin: 10px 0 5px 0;">WordPress Cron draait alleen wanneer er een bezoeker komt. Voor betrouwbare timing op exact het ingestelde tijdstip, gebruik een echte cron job:</p>
                                    <p style="margin: 5px 0;"><strong>Voor cPanel:</strong></p>
                                    <code style="display: block; background: #fff; padding: 10px; margin: 5px 0; border: 1px solid #ddd; font-size: 12px;">
*/15 * * * * wget -q -O - <?php echo esc_url(home_url('/wp-cron.php?doing_wp_cron')); ?> >/dev/null 2>&1
                                    </code>
                                    <p style="margin: 10px 0 5px 0;"><strong>Voor SSH (crontab -e):</strong></p>
                                    <code style="display: block; background: #fff; padding: 10px; margin: 5px 0; border: 1px solid #ddd; font-size: 12px;">
*/15 * * * * curl -s <?php echo esc_url(home_url('/wp-cron.php?doing_wp_cron')); ?> >/dev/null 2>&1
                                    </code>
                                    <p style="margin: 10px 0 0 0; font-size: 12px; color: #666;">Dit controleert elke 15 minuten of er geplande taken zijn. Kopieer de regel hierboven en plak deze in je cron job configuratie.</p>
                                </div>
                            <?php else: ?>
                                <em style="color: #999;">Uitgeschakeld</em>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Handmatige verzending</th>
                        <td>
                            <form method="post" action="<?php echo esc_url(admin_url('admin.php?page=bpac-settings')); ?>" style="display: inline-block; margin-top: 10px;">
                                <?php wp_nonce_field('bpac_send_now'); ?>
                                <label style="display: block; margin-bottom: 10px;">
                                    <input type="checkbox" name="bpac_ignore_already_sent" value="1" />
                                    Herverzend ook e-mails die vandaag al zijn verzonden
                                </label>
                                <p class="description" style="margin-bottom: 10px;">
                                    Verzend toegangscodes direct voor alle relevante boekingen (gebaseerd op de "Verzend de dag voor de boeking" instelling). 
                                    Dit negeert de geplande tijd en de "Automatische verzending" instelling. Handig voor test doeleinden.
                                </p>
                                <?php submit_button('Verzend codes nu', 'secondary', 'bpac_send_now', false); ?>
                            </form>
                        </td>
                    </tr>
                </table>
                
                <h3>E-mail Template</h3>
                <div class="bpac-replacement-help">
                    <strong>Beschikbare replacement velden:</strong><br>
                    <code>{naam}</code> - Klant naam<br>
                    <code>{ruimte}</code> - Ruimte/service naam<br>
                    <code>{datum}</code> - Boekingsdatum (dd-mm-yyyy)<br>
                    <code>{tijd}</code> - Boekingstijd (uu:mm)<br>
                    <code>{code}</code> - Toegangscode
                </div>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">E-mail Onderwerp</th>
                        <td>
                            <input type="text" name="bpac_email_subject" value="<?php echo esc_attr($email_subject); ?>" class="regular-text" />
                            <p class="description">Onderwerp van de e-mail. Gebruik {ruimte} voor de ruimte naam.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">E-mail Inhoud</th>
                        <td>
                            <textarea name="bpac_email_template" class="bpac-email-template-editor" rows="10"><?php echo esc_textarea($email_template); ?></textarea>
                            <p class="description">Inhoud van de e-mail. Gebruik de replacement velden hierboven.</p>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button('Instellingen opslaan', 'primary', 'bpac_save_settings'); ?>
            </form>
            
            <!-- Test Email -->
            <h3>Test E-mail</h3>
            <form method="post" action="<?php echo esc_url(admin_url('admin.php?page=bpac-settings')); ?>" style="margin-top: 20px;">
                <?php wp_nonce_field('bpac_test_email'); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row">E-mailadres</th>
                        <td>
                            <input type="email" name="test_email" value="<?php echo esc_attr(wp_get_current_user()->user_email); ?>" required />
                            <p class="description">Verstuur een test e-mail met toegangscode.</p>
                        </td>
                    </tr>
                </table>
                <?php submit_button('Verstuur test e-mail', 'secondary', 'bpac_test_email'); ?>
            </form>
        </div>
        
        <!-- Sync Rooms -->
        <div class="card" style="margin-top: 20px;">
            <h2>Ruimtes Synchroniseren</h2>
            <p>Synchroniseer ruimtes uit BookingPress. Nieuwe ruimtes worden automatisch toegevoegd.</p>
            <form method="post" action="<?php echo esc_url(admin_url('admin.php?page=bpac-settings')); ?>">
                <?php wp_nonce_field('bpac_sync_rooms'); ?>
                <?php submit_button('Synchroniseer Ruimtes', 'secondary', 'bpac_sync_rooms'); ?>
            </form>
        </div>
        
        <!-- Rooms Management -->
        <div class="card" style="margin-top: 20px;">
            <h2>Ruimtes en Toegangscodes</h2>
            
            <?php if (empty($rooms)): ?>
                <p>Geen ruimtes gevonden. Klik op "Synchroniseer Ruimtes" om ruimtes uit BookingPress te halen.</p>
            <?php else: ?>
                <form method="post" action="<?php echo esc_url(admin_url('admin.php?page=bpac-settings')); ?>">
                    <?php wp_nonce_field('bpac_update_all_codes'); ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th style="width: 30%;">Ruimte</th>
                                <th style="width: 20%;">Identifier</th>
                                <th style="width: 30%;">Toegangscode</th>
                                <th style="width: 20%;">Laatst bijgewerkt</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($rooms as $room): 
                                // Get the value from the database
                                $current_code = '';
                                if (isset($room['access_code'])) {
                                    $current_code = ($room['access_code'] !== null && $room['access_code'] !== '') 
                                        ? trim((string)$room['access_code']) 
                                        : '';
                                }
                            ?>
                                <tr>
                                    <td><strong><?php echo esc_html($room['room_name']); ?></strong></td>
                                    <td><code><?php echo esc_html($room['room_identifier']); ?></code></td>
                                    <td>
                                        <input type="text" 
                                               name="room_codes[<?php echo esc_attr($room['id']); ?>]" 
                                               value="<?php echo esc_attr($current_code); ?>" 
                                               placeholder="Geen code" 
                                               style="width: 150px;" />
                                    </td>
                                    <td>
                                        <?php if (!empty($room['last_updated'])): ?>
                                            <small><?php echo esc_html(date('d-m-Y H:i', strtotime($room['last_updated']))); ?></small>
                                        <?php else: ?>
                                            <small style="color: #999;">Nog niet bijgewerkt</small>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    <p style="margin-top: 15px;">
                        <input type="hidden" name="bpac_update_all_codes" value="1" />
                        <?php submit_button('Alle codes opslaan', 'primary', 'submit', false); ?>
                    </p>
                </form>
            <?php endif; ?>
        </div>
        
        <!-- Email Log (optional) -->
        <div class="card" style="margin-top: 20px;">
            <h2>E-mail Log (Laatste 10)</h2>
            <?php
            global $wpdb;
            $log_table = $wpdb->prefix . 'bpac_email_log';
            $logs = $wpdb->get_results(
                "SELECT * FROM $log_table ORDER BY sent_at DESC LIMIT 10",
                ARRAY_A
            );
            
            if (empty($logs)): ?>
                <p>Nog geen e-mails verzonden.</p>
            <?php else: ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th>Datum/Tijd</th>
                            <th>E-mail</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($logs as $log): ?>
                            <tr>
                                <td><?php echo esc_html($log['sent_at']); ?></td>
                                <td><?php echo esc_html($log['customer_email']); ?></td>
                                <td>
                                    <span style="color: <?php echo $log['status'] === 'sent' ? 'green' : 'red'; ?>;">
                                        <?php echo esc_html($log['status'] === 'sent' ? '✓ Verzonden' : '✗ Mislukt'); ?>
                                    </span>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>
    <?php
}

