<?php
/*
Plugin Name: BookingPress Access Codes
Description: Verstuurt automatisch toegangscodes voor sleutelkluisjes naar klanten met BookingPress boekingen.
Version: 1.0.20
Author: vanMeerdervoort
*/

if (!defined('ABSPATH')) exit;

// Plugin constants
define('BPAC_VERSION', '1.0.20');
define('BPAC_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('BPAC_PLUGIN_URL', plugin_dir_url(__FILE__));
define('BPAC_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Include required files
$includes_dir = BPAC_PLUGIN_DIR . 'includes/';
if (file_exists($includes_dir . 'database-check.php')) {
    require_once $includes_dir . 'database-check.php';
}
if (file_exists($includes_dir . 'room-manager.php')) {
    require_once $includes_dir . 'room-manager.php';
}
if (file_exists($includes_dir . 'email-handler.php')) {
    require_once $includes_dir . 'email-handler.php';
}
if (file_exists($includes_dir . 'admin-settings.php')) {
    require_once $includes_dir . 'admin-settings.php';
}

// Activation/Deactivation hooks
register_activation_hook(__FILE__, 'bpac_activate');
register_deactivation_hook(__FILE__, 'bpac_deactivate');

function bpac_activate() {
    global $wpdb;
    
    // Create database tables
    $charset_collate = $wpdb->get_charset_collate();
    
    // Rooms table
    $table_rooms = $wpdb->prefix . 'bpac_rooms';
    $sql_rooms = "CREATE TABLE IF NOT EXISTS $table_rooms (
        id INT AUTO_INCREMENT PRIMARY KEY,
        room_identifier VARCHAR(255) NOT NULL UNIQUE,
        room_name VARCHAR(255) NOT NULL,
        access_code VARCHAR(50) DEFAULT NULL,
        last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    ) $charset_collate;";
    
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql_rooms);
    
    // Email log table
    $table_log = $wpdb->prefix . 'bpac_email_log';
    $sql_log = "CREATE TABLE IF NOT EXISTS $table_log (
        id INT AUTO_INCREMENT PRIMARY KEY,
        booking_id BIGINT(11) DEFAULT NULL,
        room_id INT DEFAULT NULL,
        customer_email VARCHAR(255) DEFAULT NULL,
        sent_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        status VARCHAR(20) DEFAULT 'sent'
    ) $charset_collate;";
    
    dbDelta($sql_log);
    
    // Set default options
    if (get_option('bpac_daily_send_time') === false) {
        add_option('bpac_daily_send_time', '08:00');
    }
    if (get_option('bpac_enable_auto_send') === false) {
        add_option('bpac_enable_auto_send', 1);
    }
    if (get_option('bpac_send_day_before') === false) {
        add_option('bpac_send_day_before', 0);
    }
    
    // Schedule cron job
    if (function_exists('bpac_activate_cron')) {
        bpac_activate_cron();
    }
}

function bpac_deactivate() {
    // Clear cron job
    wp_clear_scheduled_hook('bpac_daily_send_codes');
}

// Cron job activation
function bpac_activate_cron() {
    // Only schedule if auto send is enabled
    $enable_auto_send = get_option('bpac_enable_auto_send', 1);
    if (!$enable_auto_send) {
        return;
    }
    
    $send_time = get_option('bpac_daily_send_time', '08:00');
    $time_parts = explode(':', $send_time);
    $hour = intval($time_parts[0]);
    $minute = isset($time_parts[1]) ? intval($time_parts[1]) : 0;
    
    $timestamp = strtotime("today $hour:$minute");
    if ($timestamp < time()) {
        $timestamp = strtotime("tomorrow $hour:$minute");
    }
    
    if (!wp_next_scheduled('bpac_daily_send_codes')) {
        wp_schedule_event($timestamp, 'daily', 'bpac_daily_send_codes');
    }
}

// Cron action
add_action('bpac_daily_send_codes', 'bpac_send_daily_codes');

function bpac_send_daily_codes() {
    if (!function_exists('bpac_send_access_codes_for_today')) {
        require_once BPAC_PLUGIN_DIR . 'includes/email-handler.php';
    }
    bpac_send_access_codes_for_today();
}

// Update system (like Simple Calendar Importer)
add_filter('pre_set_site_transient_update_plugins', 'bpac_check_for_updates');
function bpac_check_for_updates($transient) {
    if (empty($transient->checked)) return $transient;
    
    $remote = wp_remote_get('https://plugins.nietstezienhier.nl/bookingpress-access-codes/update.json', [
        'timeout' => 15,
        'sslverify' => false,
    ]);
    
    if (is_wp_error($remote) || wp_remote_retrieve_response_code($remote) !== 200) {
        return $transient;
    }
    
    $plugin_data = json_decode(wp_remote_retrieve_body($remote));
    if (!$plugin_data) return $transient;
    
    $plugin_file = BPAC_PLUGIN_BASENAME;
    $current_version = $transient->checked[$plugin_file] ?? null;
    
    if ($current_version && version_compare($plugin_data->version, $current_version, '>')) {
        $transient->response[$plugin_file] = (object)[
            'slug' => $plugin_data->slug,
            'new_version' => $plugin_data->version,
            'url' => $plugin_data->homepage,
            'package' => $plugin_data->download_url,
        ];
    }
    
    return $transient;
}

add_filter('plugins_api', 'bpac_plugin_info', 10, 3);
function bpac_plugin_info($result, $action, $args) {
    if ($action !== 'plugin_information' || $args->slug !== 'bookingpress-access-codes') {
        return $result;
    }
    
    $remote = wp_remote_get('https://plugins.nietstezienhier.nl/bookingpress-access-codes/update.json', [
        'timeout' => 15,
        'sslverify' => false,
    ]);
    
    if (is_wp_error($remote) || wp_remote_retrieve_response_code($remote) !== 200) {
        return $result;
    }
    
    $plugin_data = json_decode(wp_remote_retrieve_body($remote));
    if (!$plugin_data) return $result;
    
    return (object)[
        'name' => $plugin_data->name,
        'slug' => $plugin_data->slug,
        'version' => $plugin_data->version,
        'author' => $plugin_data->author,
        'homepage' => $plugin_data->homepage,
        'sections' => (array) $plugin_data->sections,
        'download_link' => $plugin_data->download_url,
    ];
}

